<?php
// panel/config.php

declare(strict_types=1);

// --------------------
// ENV helper (optional .env support)
// --------------------
function env(string $key, mixed $default = null): mixed {
    $val = getenv($key);
    if ($val === false) {
        $val = $_ENV[$key] ?? $_SERVER[$key] ?? null;
    }
    return ($val === null || $val === '') ? $default : $val;
}

$envFile = __DIR__ . '/.env';
if (is_file($envFile) && is_readable($envFile)) {
    foreach (file($envFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES) as $line) {
        $line = trim($line);
        if ($line === '' || str_starts_with($line, '#') || !str_contains($line, '=')) continue;
        [$k, $v] = array_map('trim', explode('=', $line, 2));
        $v = trim($v, "\"'");
        if ($k !== '' && getenv($k) === false) putenv($k . '=' . $v);
    }
}

// --------------------
// App settings
// --------------------
define('APP_ENV', (string) env('APP_ENV', 'prod')); // prod|dev
define('APP_DEBUG', APP_ENV !== 'prod');

date_default_timezone_set((string) env('APP_TZ', 'Europe/Istanbul'));

if (APP_DEBUG) {
    ini_set('display_errors', '1');
    ini_set('display_startup_errors', '1');
    error_reporting(E_ALL);
} else {
    ini_set('display_errors', '0');
    ini_set('display_startup_errors', '0');
}

// --------------------
// Paths
// --------------------
define('PANEL_PATH', realpath(__DIR__));
define('ROOT_PATH', realpath(__DIR__ . '/..'));
define('UPLOAD_PATH', ROOT_PATH . '/uploads'); // örn: /uploads/tesis, /uploads/ilan, /uploads/medya

// --------------------
// URLs
// --------------------
$scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
$host = $_SERVER['HTTP_HOST'] ?? (string) env('APP_HOST', 'www.bungalovsepeti.com.tr');

define('BASE_URL', rtrim((string) env('BASE_URL', $scheme . '://' . $host), '/'));
define('PANEL_URL', BASE_URL . '/panel');

// --------------------
// DB (PDO)
// --------------------
define('DB_HOST', (string) env('DB_HOST', 'localhost'));
define('DB_NAME', (string) env('DB_NAME', 'bungalowsepeti_2026'));
define('DB_USER', (string) env('DB_USER', 'bungalowsepeti_2026'));
define('DB_PASS', (string) env('DB_PASS', '3HkCMJy^Z_5S'));

function db(): PDO {
    static $pdo = null;
    if ($pdo instanceof PDO) return $pdo;

    $dsn = 'mysql:host=' . DB_HOST . ';dbname=' . DB_NAME . ';charset=utf8mb4';
    $options = [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        PDO::ATTR_EMULATE_PREPARES => false,
    ];

    $pdo = new PDO($dsn, DB_USER, DB_PASS, $options);
    return $pdo;
}

// --------------------
// Session (secure defaults)
// --------------------
$secure = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off');
session_set_cookie_params([
    'lifetime' => 0,
    'path' => '/',
    'domain' => '',
    'secure' => $secure,
    'httponly' => true,
    'samesite' => 'Lax',
]);

if (session_status() !== PHP_SESSION_ACTIVE) session_start();

// --------------------
// Common helpers
// --------------------
function e(string $str): string { return htmlspecialchars($str, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8'); }
function is_post(): bool { return ($_SERVER['REQUEST_METHOD'] ?? 'GET') === 'POST'; }

function redirect(string $pathOrUrl): never {
    $url = str_starts_with($pathOrUrl, 'http') ? $pathOrUrl : PANEL_URL . '/' . ltrim($pathOrUrl, '/');
    header('Location: ' . $url);
    exit;
}

// --------------------
// Auth helpers (rol: admin/kullanici)
// --------------------
function current_user(): ?array {
    return $_SESSION['user'] ?? null; // ['id'=>..., 'rol'=>..., 'email'=>..., 'tesis_ismi'=>...]
}

function require_login(): void {
    if (!current_user()) redirect('giris.php');
}

function require_admin(): void {
    require_login();
    if ((current_user()['rol'] ?? '') !== 'admin') redirect('index.php');
}

// --------------------
// CSRF
// --------------------
function csrf_token(): string {
    if (empty($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }
    return $_SESSION['csrf_token'];
}

function csrf_validate(?string $token): bool {
    return is_string($token) && isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $token);
}
