<?php
// /home/bungalowsepeti/public_html/panel/ilan-ekle.php
require_once __DIR__ . '/init.php';

require_login();
auth_refresh_user();

$u = current_user();
$tesisId = (int)($u['id'] ?? 0);

/* ---------------------------
   CONFIG
----------------------------*/
const ILAN_KREDI_MALIYETI = 100;
const MIN_MULK_SECIM = 3;
const MIN_TESIS_SECIM = 3;
const MIN_ODA_SECIM = 6;
const MIN_GALERI = 6;
const MAX_GALERI = 10;
const AJAX_MEDIA_LIMIT = 80;
const DIST_MAX = 1000000;

/* ---------------------------
   HELPERS
----------------------------*/
function json_out(array $data, int $code = 200): never {
  http_response_code($code);
  header('Content-Type: application/json; charset=utf-8');
  echo json_encode($data, JSON_UNESCAPED_UNICODE);
  exit;
}

function tr_slug(string $s): string {
  $s = trim(mb_strtolower($s));
  $map = ['ç' => 'c', 'ğ' => 'g', 'ı' => 'i', 'ö' => 'o', 'ş' => 's', 'ü' => 'u'];
  $s = strtr($s, $map);
  $s = preg_replace('~[^a-z0-9]+~u', '-', $s);
  $s = trim($s, '-');
  $s = preg_replace('~-+~', '-', $s);
  return $s ?: ('ilan-' . random_int(1000, 9999));
}

function pos_int($v, int $max = 99999999, int $min = 1): ?int {
  $v = trim((string)$v);
  if ($v === '') return null;
  if (!preg_match('~^\d+$~', $v)) return null;
  $n = (int)$v;
  if ($n < $min || $n > $max) return null;
  return $n;
}

function enum_0_10($v): string {
  $v = (string)$v;
  return preg_match('~^(0|1|2|3|4|5|6|7|8|9|10)$~', $v) ? $v : '0';
}

function enum_1_10($v): string {
  $v = (string)$v;
  return preg_match('~^(1|2|3|4|5|6|7|8|9|10)$~', $v) ? $v : '';
}

function has_angle(string $s): bool {
  return (bool)preg_match('~[<>]~', $s);
}

function clean_text(string $s, int $maxLen): string {
  $s = preg_replace('~[\x00-\x1F\x7F]~u', '', $s);
  $s = trim($s);
  if ($maxLen > 0) $s = mb_substr($s, 0, $maxLen);
  return $s;
}

function dist_int(string $v, int $max = DIST_MAX): ?int {
  $v = trim($v);
  if ($v === '') return null;
  $v = preg_replace('~\s+~u', '', $v);
  if (!preg_match('~^\d+$~', $v)) return null;
  $i = (int)$v;
  if ($i <= 0 || $i > $max) return null;
  return $i;
}

function rez_norm(string $v): ?string {
  $v = trim($v);
  if ($v === '') return null;
  if (has_angle($v)) return null;
  $v = preg_replace('~[\x00-\x1F\x7F]~u', '', $v);

  if (preg_match('~^https?://~i', $v)) {
    return filter_var($v, FILTER_VALIDATE_URL) ? $v : null;
  }

  $phone = preg_replace('~[^\d+]~', '', $v);
  $digits = preg_replace('~\D~', '', $phone);
  if ($digits !== '' && strlen($digits) >= 8 && strlen($digits) <= 20) return $phone;

  return null;
}
function rez_norm_whatsapp(string $v): ?string {
  $v = trim($v);
  if ($v === '') return null;
  if (has_angle($v)) return null;
  $v = preg_replace('~[\x00-\x1F\x7F]~u', '', $v);

  if (preg_match('~^https?://~i', $v)) {
    return filter_var($v, FILTER_VALIDATE_URL) ? $v : null;
  }

  $digits = preg_replace('~\D~', '', $v);

  if (preg_match('~^0\d{10}$~', $digits)) return $digits;            // 05419876543
  if (preg_match('~^90(\d{10})$~', $digits, $m)) return '0' . $m[1]; // 905419876543 -> 05419876543

  return null;
}
function public_url_from_rel(string $rel): string {
  return rtrim(BASE_URL, '/') . '/' . ltrim($rel, '/');
}

function safe_id($x): int {
  if (is_int($x)) return $x > 0 ? $x : 0;
  if (is_string($x)) {
    $t = trim($x);
    if ($t === '' || !ctype_digit($t)) return 0;
    $n = (int)$t;
    return $n > 0 ? $n : 0;
  }
  return 0;
}

function norm_id_list($raw, ?array $allowedSet = null, int $exclude = 0, int $limit = 0): array {
  $out = [];
  $seen = [];
  foreach ((array)$raw as $x) {
    $id = safe_id($x);
    if ($id <= 0) continue;
    if ($exclude > 0 && $id === $exclude) continue;
    if ($allowedSet !== null && !isset($allowedSet[$id])) continue;
    if (isset($seen[$id])) continue;
    $seen[$id] = true;
    $out[] = $id;
    if ($limit > 0 && count($out) >= $limit) break;
  }
  return $out;
}

function build_allowed_set(array $rows): array {
  $set = [];
  foreach ($rows as $r) {
    $v = null;
    if (is_array($r)) $v = $r['id'] ?? null;
    elseif (is_object($r)) $v = $r->id ?? null;
    $id = safe_id($v);
    if ($id > 0) $set[$id] = true;
  }
  return $set;
}

/* ---------------------------
   FORM SCHEMA (tek yerden mesaj/alan listesi)
----------------------------*/
$formSchema = [
  'rules' => [
    'krediMaliyet' => (int)ILAN_KREDI_MALIYETI,
    'minMulk' => (int)MIN_MULK_SECIM,
    'minTesis' => (int)MIN_TESIS_SECIM,
    'minOda' => (int)MIN_ODA_SECIM,
    'minGaleri' => (int)MIN_GALERI,
    'maxGaleri' => (int)MAX_GALERI,
    'distMax' => (int)DIST_MAX,
  ],
  'sectionMessages' => [
    'category' => 'İlan Kategorisi seçmelisin.',
    'features' => 'İlan Özellikleri: zorunlu alanları doğru formatta doldurmalısın.',
    'mulks' => 'Mülk Tipleri: en az ' . (int)MIN_MULK_SECIM . ' seçim yapmalısın.',
    'tesis' => 'Tesis Özellikleri: en az ' . (int)MIN_TESIS_SECIM . ' seçim yapmalısın.',
    'oda' => 'Oda Özellikleri: en az ' . (int)MIN_ODA_SECIM . ' seçim yapmalısın.',
    'media' => 'İlan Görselleri: 1 kapak + en az ' . (int)MIN_GALERI . ' galeri görseli seçmelisin.',
    'distance' => 'Mesafe Haritası: tüm alanları doldurmalı ve sayısal alanlara geçerli değer girmelisin.',
    'rez' => 'Rezervasyon Linkleri: en az 1 adet geçerli link girmelisin.',
  ],
  'distanceFields' => [
    ['name' => 'havaalani_mesafe', 'required' => 'Havalimanı mesafesi zorunlu.'],
    ['name' => 'plaj_mesafe', 'required' => 'Plaj mesafesi zorunlu.'],
    ['name' => 'otogar_mesafe', 'required' => 'Otogar mesafesi zorunlu.'],
    ['name' => 'market_mesafe', 'required' => 'Market mesafesi zorunlu.'],
    ['name' => 'sehir_merkezi_mesafe', 'required' => 'Şehir merkezi mesafesi zorunlu.'],
    ['name' => 'restaurant_mesafe', 'required' => 'Restaurant mesafesi zorunlu.'],
  ],
];

/* ---------------------------
   AJAX: ilan ekle -> medya liste
----------------------------*/
if (is_post() && (string)($_POST['action'] ?? '') === 'ajax_media_list') {
  if (!csrf_validate($_POST['csrf_token'] ?? null)) {
    json_out(['ok' => false, 'message' => 'Güvenlik doğrulaması başarısız.'], 403);
  }

  $st = db()->prepare(
    "SELECT id, dosya_adi, dosya_yol, boyut
       FROM tesis_medya
      WHERE tesis_id = :t
      ORDER BY sort_key DESC, id DESC
      LIMIT " . (int)AJAX_MEDIA_LIMIT
  );
  $st->execute([':t' => $tesisId]);

  $rows = $st->fetchAll();
  $items = [];
  foreach ($rows as $r) {
    $rel = (string)($r['dosya_yol'] ?? '');
    $items[] = [
      'id' => safe_id($r['id'] ?? null),
      'file' => (string)($r['dosya_adi'] ?? ''),
      'rel' => $rel,
      'url' => public_url_from_rel($rel),
      'size' => (int)($r['boyut'] ?? 0),
    ];
  }

  json_out(['ok' => true, 'items' => $items]);
}

/* ---------------------------   DATA  ----------------------------*/
$havaalanlari = [
  'Adana Çukurova Uluslararası Havalimanı',
  'Adıyaman Havalimanı',
  'Ağrı Ahmed-i Hani Havalimanı',
  'Amasya Merzifon Havalimanı',
  'Ankara Esenboğa Havalimanı',
  'Antalya Havalimanı',
  'Balıkesir Koca Seyit Havalimanı',
  'Batman Havalimanı',
  'Bingöl Havalimanı',
  'Bursa Yenişehir Havalimanı',
  'Çanakkale Havalimanı',
  'Çanakkale Gökçeada Havalimanı',
  'Denizli Çardak Havalimanı',
  'Diyarbakır Havalimanı',
  'Elazığ Havalimanı',
  'Erzincan Havalimanı',
  'Erzurum Havalimanı',
  'Eskişehir Hasan Polatkan Havalimanı',
  'Gaziantep Havalimanı',
  'Gazipaşa-Alanya Havalimanı',
  'Giresun-Ordu Havalimanı',
  'Hakkari Yüksekova Selahaddin Eyyubi Havalimanı',
  'Hatay Havalimanı',
  'Iğdır Şehit Bülent Aydın Havalimanı',
  'Isparta Süleyman Demirel Havalimanı',
  'İstanbul Havalimanı',
  'İstanbul Sabiha Gökçen Havalimanı',
  'İzmir Adnan Menderes Havalimanı',
  'Kahramanmaraş Havalimanı',
  'Kars Harakani Havalimanı',
  'Kastamonu Havalimanı',
  'Kayseri Erkilet Havalimanı',
  'Kocaeli Cengiz Topel Havalimanı',
  'Konya Havalimanı',
  'Kütahya Zafer Havalimanı',
  'Malatya Havalimanı',
  'Mardin Havalimanı',
  'Muğla Dalaman Havalimanı',
  'Muğla Milas-Bodrum Havalimanı',
  'Muş Sultan Alparslan Havalimanı',
  'Nevşehir Kapadokya Havalimanı',
  'Ordu-Giresun Havalimanı',
  'Rize-Artvin Havalimanı',
  'Samsun Çarşamba Havalimanı',
  'Siirt Havalimanı',
  'Sinop Havalimanı',
  'Sivas Nuri Demirağ Havalimanı',
  'Şanlıurfa GAP Havalimanı',
  'Şırnak Şerafettin Elçi Havalimanı',
  'Tekirdağ Çorlu Atatürk Havalimanı',
  'Tokat Havalimanı',
  'Trabzon Havalimanı',
  'Uşak Havalimanı',
  'Van Ferit Melen Havalimanı',
  'Zonguldak Çaycuma Havalimanı',
];
$allowedHavaalanlari = array_fill_keys($havaalanlari, true);

$mulks  = db()->query("SELECT id, isim FROM mulktipleri ORDER BY sirano ASC, id ASC")->fetchAll();
$odas   = db()->query("SELECT id, isim FROM oda_ozellikleri ORDER BY sirano ASC, id ASC")->fetchAll();
$tesiss = db()->query("SELECT id, isim FROM tesis_ozellikleri ORDER BY sirano ASC, id ASC")->fetchAll();
// sirano yeterli; sira kolonu tekrarlı kullanım kokusu yaratıyordu
$plats  = db()->query("SELECT id, isim FROM rezervasyon_platformlari ORDER BY sirano ASC, id ASC")->fetchAll();
$kats   = db()->query("SELECT id, isim FROM tesis_kategori ORDER BY id ASC")->fetchAll();

$allowedMulks = build_allowed_set($mulks);
$allowedOdas  = build_allowed_set($odas);
$allowedTesis = build_allowed_set($tesiss);
$allowedPlats = build_allowed_set($plats);
$allowedKats  = build_allowed_set($kats);

$error = null;

$val = [
  'ilan_turu_id' => '',
  'oda_ismi' => '',
  'fiyat' => '',
  'indirimli_fiyat' => '',
  'max_yetiskin' => '',
  'max_cocuk' => '',
  'min_konaklama' => '',
  'yatak_odasi' => '',
  'yatak_sayisi' => '',
  'banyo' => '',
  'havuz' => '',
  'evlilik_sarti' => '',
  'metrekare' => '',
  'aciklama' => '',
  'havaalani_ismi' => '',
  'plaj_mesafe' => '',
  'otogar_mesafe' => '',
  'market_mesafe' => '',
  'havaalani_mesafe' => '',
  'sehir_merkezi_mesafe' => '',
  'restaurant_mesafe' => '',
  'hava_mesafe_birim' => 'km',
  'plaj_mesafe_birim' => 'km',
  'otogar_birim' => 'km',
  'market_mesafe_birim' => 'km',
  'sehir_mesafe_birim' => 'km',
  'rest_mesafe_birim' => 'km',
  'mulktipleri' => [],
  'oda_ozellikleri' => [],
  'tesis_ozellikleri' => [],
  'rez' => [],
  'cover_media_id' => '',
  'gallery_media_ids' => '',
];

/* ---------------------------
   POST: save
----------------------------*/
if (is_post() && (string)($_POST['action'] ?? '') === '') {
  if (!csrf_validate($_POST['csrf_token'] ?? null)) $error = 'Güvenlik doğrulaması başarısız.';

  $saveAsDraft  = ((string)($_POST['save_as_draft'] ?? '') === '1');
  $sozlesmeOnay = ((string)($_POST['sozlesme_onay'] ?? '') === '1');

  $val['ilan_turu_id'] = (string)($_POST['ilan_turu_id'] ?? '');
  $val['oda_ismi'] = clean_text((string)($_POST['oda_ismi'] ?? ''), 255);
  $val['aciklama'] = clean_text((string)($_POST['aciklama'] ?? ''), 5000);

  $val['fiyat'] = (string)($_POST['fiyat'] ?? '');
  $val['indirimli_fiyat'] = (string)($_POST['indirimli_fiyat'] ?? '');

  $tmp = trim((string)($_POST['max_yetiskin'] ?? ''));
  $val['max_yetiskin'] = ($tmp === '') ? '' : enum_1_10($tmp);

  $tmp = trim((string)($_POST['max_cocuk'] ?? ''));
  $val['max_cocuk'] = ($tmp === '') ? '' : enum_0_10($tmp);

  $val['min_konaklama'] = (string)($_POST['min_konaklama'] ?? '');

  $tmp = trim((string)($_POST['yatak_odasi'] ?? ''));
  $val['yatak_odasi'] = ($tmp === '') ? '' : enum_0_10($tmp);

  $val['yatak_sayisi'] = (string)($_POST['yatak_sayisi'] ?? '');

  $tmp = trim((string)($_POST['banyo'] ?? ''));
  $val['banyo'] = ($tmp === '') ? '' : enum_0_10($tmp);

  $tmp = trim((string)($_POST['havuz'] ?? ''));
  $val['havuz'] = ($tmp === 'Var' || $tmp === 'Yok') ? $tmp : '';

  $tmp = trim((string)($_POST['evlilik_sarti'] ?? ''));
  $val['evlilik_sarti'] = ($tmp === 'Var' || $tmp === 'Yok') ? $tmp : '';

  $val['metrekare'] = (string)($_POST['metrekare'] ?? '');

  $val['havaalani_ismi'] = clean_text((string)($_POST['havaalani_ismi'] ?? ''), 255);

  $val['plaj_mesafe'] = (string)($_POST['plaj_mesafe'] ?? '');
  $val['otogar_mesafe'] = (string)($_POST['otogar_mesafe'] ?? '');
  $val['market_mesafe'] = (string)($_POST['market_mesafe'] ?? '');
  $val['havaalani_mesafe'] = (string)($_POST['havaalani_mesafe'] ?? '');
  $val['sehir_merkezi_mesafe'] = (string)($_POST['sehir_merkezi_mesafe'] ?? '');
  $val['restaurant_mesafe'] = (string)($_POST['restaurant_mesafe'] ?? '');

  $units = ['km', 'm'];
  $val['hava_mesafe_birim'] = in_array((string)($_POST['hava_mesafe_birim'] ?? 'km'), $units, true) ? (string)$_POST['hava_mesafe_birim'] : 'km';
  $val['plaj_mesafe_birim'] = in_array((string)($_POST['plaj_mesafe_birim'] ?? 'km'), $units, true) ? (string)$_POST['plaj_mesafe_birim'] : 'km';
  $val['otogar_birim'] = in_array((string)($_POST['otogar_birim'] ?? 'km'), $units, true) ? (string)$_POST['otogar_birim'] : 'km';
  $val['market_mesafe_birim'] = in_array((string)($_POST['market_mesafe_birim'] ?? 'km'), $units, true) ? (string)$_POST['market_mesafe_birim'] : 'km';
  $val['sehir_mesafe_birim'] = in_array((string)($_POST['sehir_mesafe_birim'] ?? 'km'), $units, true) ? (string)$_POST['sehir_mesafe_birim'] : 'km';
  $val['rest_mesafe_birim'] = in_array((string)($_POST['rest_mesafe_birim'] ?? 'km'), $units, true) ? (string)$_POST['rest_mesafe_birim'] : 'km';

  $val['cover_media_id'] = trim((string)($_POST['cover_media_id'] ?? ''));
  $val['gallery_media_ids'] = trim((string)($_POST['gallery_media_ids'] ?? ''));

  $val['mulktipleri'] = norm_id_list((array)($_POST['mulktipleri'] ?? []), $allowedMulks);
  $val['oda_ozellikleri'] = norm_id_list((array)($_POST['oda_ozellikleri'] ?? []), $allowedOdas);
  $val['tesis_ozellikleri'] = norm_id_list((array)($_POST['tesis_ozellikleri'] ?? []), $allowedTesis);

  $rez = (array)($_POST['rez'] ?? []);
  $val['rez'] = $rez;

  if (!$error && !isset($allowedKats[safe_id($val['ilan_turu_id'])])) $error = 'Kategori seçimi hatalı.';
  if (!$error && $val['oda_ismi'] === '') $error = 'Oda ismi zorunlu.';
  if (!$error && $val['aciklama'] === '') $error = 'İlan açıklaması zorunlu.';

  // text alanlarda < > engeli (havaalani_ismi zaten allow-list ile doğrulanıyor)
  if (!$error) {
    foreach (['oda_ismi', 'aciklama'] as $k) {
      if (has_angle((string)$val[$k])) { $error = 'Geçersiz karakter.'; break; }
    }
  }

  if (!$error && trim((string)$val['havaalani_ismi']) === '') $error = 'En yakın havalimanı ismi zorunlu.';
  if (!$error && $val['havaalani_ismi'] !== '' && !isset($allowedHavaalanlari[$val['havaalani_ismi']])) $error = 'Havalimanı seçimi hatalı.';

  $fiyat = pos_int($val['fiyat']);
  if (!$error && $fiyat === null) $error = 'Fiyat zorunlu.';

  $ind = pos_int($val['indirimli_fiyat']);
  if (!$error && $ind !== null && $fiyat !== null && $ind >= $fiyat) $error = 'İndirimli fiyat, fiyattan küçük olmalı.';

  if (!$error && $val['max_yetiskin'] === '') $error = 'Maks. yetişkin seç.';
  if (!$error && $val['max_cocuk'] === '') $error = 'Maks. çocuk seç.';

  $minKon = pos_int($val['min_konaklama'], 255, 1);
  if (!$error && $minKon === null) $error = 'Minimum konaklama zorunlu.';

  $kisiToplam = (int)$val['max_yetiskin'] + (int)$val['max_cocuk'];
  if (!$error && $kisiToplam <= 0) $error = 'Kapasite hatalı.';

  if (!$error && $val['yatak_odasi'] === '') $error = 'Yatak odası seç.';

  $yatak_sayisi = pos_int($val['yatak_sayisi'], 20, 1);
  if (!$error && $yatak_sayisi === null) $error = 'Yatak sayısı seç.';

  if (!$error && $val['banyo'] === '') $error = 'Banyo sayısı seç.';
  if (!$error && $val['havuz'] === '') $error = 'Havuz seç.';
  if (!$error && $val['evlilik_sarti'] === '') $error = 'Evlilik şartı seç.';

  $metrekare = pos_int($val['metrekare']);
  if (!$error && $metrekare === null) $error = 'm² bilgisi zorunlu.';

  $credit = (int)($u['kredi_puani'] ?? 0);
  if (!$error && !$saveAsDraft && !$sozlesmeOnay) $error = 'İlanı göndermek için sözleşmeyi onaylamalısın.';
  if (!$error && !$saveAsDraft && $credit < ILAN_KREDI_MALIYETI) $error = 'İlan eklemek için en az 100 kredi gerekir.';

  if (!$error && count($val['mulktipleri']) < MIN_MULK_SECIM) $error = 'Mülk Tipleri: en az 3 seçim yap.';
  if (!$error && count($val['tesis_ozellikleri']) < MIN_TESIS_SECIM) $error = 'Tesis Özellikleri: en az 3 seçim yap.';
  if (!$error && count($val['oda_ozellikleri']) < MIN_ODA_SECIM) $error = 'Oda Özellikleri: en az 6 seçim yap.';

  if (!$error) {
    foreach ($formSchema['distanceFields'] as $row) {
      $k = (string)$row['name'];
      $msg = (string)$row['required'];
      $n = dist_int((string)($val[$k] ?? ''), DIST_MAX);
      if ($n === null) { $error = $msg; break; }
      $val[$k] = $n;
    }
  }

  $rezClean = [];
  if (!$error) {
    foreach ($plats as $p) {
      $pid = safe_id($p['id'] ?? null);
      if ($pid <= 0 || !isset($allowedPlats[$pid])) continue;

      $raw = trim((string)($rez[$pid] ?? ''));
      if ($raw === '') continue;

      $clean = rez_norm($raw);
      if ($clean === null) { $error = 'Rezervasyon linki hatalı.'; break; }
      $rezClean[$pid] = $clean;
    }
    if (!$error && count($rezClean) < 1) $error = 'Rezervasyon Linkleri: en az 1 link gir.';
  }

  $coverId = safe_id($val['cover_media_id']);

  $galRaw = [];
  if ($val['gallery_media_ids'] !== '') $galRaw = explode(',', $val['gallery_media_ids']);
  $galIds = norm_id_list($galRaw, null, $coverId, MAX_GALERI);

  if (!$error && $coverId <= 0) $error = 'Kapak görseli seç.';
  if (!$error && count($galIds) < MIN_GALERI) $error = 'İlan Görselleri: galeride en az 6 görsel seç.';

  $mediaIds = norm_id_list(array_merge([$coverId], $galIds));

  $map = [];
if (!$error) {
  $ph = implode(',', array_fill(0, count($mediaIds), '?'));
  $stM = db()->prepare("SELECT id, dosya_yol FROM tesis_medya WHERE tesis_id = ? AND id IN ($ph)");
  $stM->execute(array_merge([$tesisId], $mediaIds));
  $rowsM = $stM->fetchAll();

  foreach ($rowsM as $r) {
    $mid = safe_id($r['id'] ?? null);
    if ($mid > 0) $map[$mid] = (string)($r['dosya_yol'] ?? '');
  }

  if (!isset($map[$coverId])) $error = 'Kapak görseli bulunamadı.';

  if (!$error) {
    foreach ($galIds as $gid) {
      if (!isset($map[$gid])) { $error = 'Galeri görseli bulunamadı.'; break; }
    }
  }
}

if (!$error) {
  $coverRel = $map[$coverId];
  $yayinDurumu = $saveAsDraft ? 'taslak' : 'incelemede';

  db()->beginTransaction();
  try {
    $krediDusuldu = false;

    if (!$saveAsDraft) {
      $stC = db()->prepare("
        UPDATE tesisler
           SET kredi_puani = kredi_puani - :k1
         WHERE id = :t
           AND kredi_puani >= :k2
      ");
      $stC->execute([
        ':t'  => $tesisId,
        ':k1' => ILAN_KREDI_MALIYETI,
        ':k2' => ILAN_KREDI_MALIYETI,
      ]);
      if ($stC->rowCount() !== 1) throw new RuntimeException('Kredi yetersiz.');
      $krediDusuldu = true;
    }

    $baseSlug = tr_slug($val['oda_ismi']);
    $slug = '';

    $stI = db()->prepare("
      INSERT INTO ilanlar
              (tesis_id, oda_ismi, url, fiyat, indirimli_fiyat,
               min_konaklama, max_yetiskin, max_cocuk,
               yatak_odasi, yatak_sayisi, banyo, havuz, aciklama,
               havaalani_ismi, plaj_mesafe, otogar_mesafe, market_mesafe, havaalani_mesafe, sehir_merkezi_mesafe, restaurant_mesafe,
               ilan_turu_id, evlilik_sarti, metrekare, kapak_gorsel, yayin_durumu, durum,
               hava_mesafe_birim, plaj_mesafe_birim, otogar_birim, market_mesafe_birim, sehir_mesafe_birim, rest_mesafe_birim)
        VALUES
              (:tesis_id, :oda_ismi, :url, :fiyat, :ind,
               :min_konaklama, :max_yetiskin, :max_cocuk,
               :yod, :ys, :banyo, :havuz, :aciklama,
               :havaalani_ismi, :plaj_mesafe, :otogar_mesafe, :market_mesafe, :havaalani_mesafe, :sehir_merkezi_mesafe, :restaurant_mesafe,
               :ilan_turu_id, :evlilik_sarti, :metrekare, :kapak_gorsel, :yayin_durumu, :durum,
               :hava_mesafe_birim, :plaj_mesafe_birim, :otogar_birim, :market_mesafe_birim, :sehir_mesafe_birim, :rest_mesafe_birim)
    ");

    $insertPayload = [
      ':tesis_id' => $tesisId,
      ':oda_ismi' => $val['oda_ismi'],
      ':fiyat' => $fiyat,
      ':ind' => $ind,
      ':min_konaklama' => $minKon,
      ':max_yetiskin' => (int)$val['max_yetiskin'],
      ':max_cocuk' => (int)$val['max_cocuk'],
      ':yod' => (string)$val['yatak_odasi'],
      ':ys' => $yatak_sayisi,
      ':banyo' => (string)$val['banyo'],
      ':havuz' => $val['havuz'],
      ':aciklama' => $val['aciklama'],
      ':havaalani_ismi' => $val['havaalani_ismi'],
      ':plaj_mesafe' => (int)$val['plaj_mesafe'],
      ':otogar_mesafe' => (int)$val['otogar_mesafe'],
      ':market_mesafe' => (int)$val['market_mesafe'],
      ':havaalani_mesafe' => (int)$val['havaalani_mesafe'],
      ':sehir_merkezi_mesafe' => (int)$val['sehir_merkezi_mesafe'],
      ':restaurant_mesafe' => (int)$val['restaurant_mesafe'],
      ':ilan_turu_id' => safe_id($val['ilan_turu_id']),
      ':evlilik_sarti' => $val['evlilik_sarti'],
      ':metrekare' => $metrekare,
      ':kapak_gorsel' => $coverRel,
      ':yayin_durumu' => $yayinDurumu,
      ':durum' => '0',
      ':hava_mesafe_birim' => $val['hava_mesafe_birim'],
      ':plaj_mesafe_birim' => $val['plaj_mesafe_birim'],
      ':otogar_birim' => $val['otogar_birim'],
      ':market_mesafe_birim' => $val['market_mesafe_birim'],
      ':sehir_mesafe_birim' => $val['sehir_mesafe_birim'],
      ':rest_mesafe_birim' => $val['rest_mesafe_birim'],
    ];

    for ($tries = 0; $tries < 20; $tries++) {
      $candidate = $baseSlug . ($tries === 0 ? '' : '-' . random_int(10, 99));
      try {
        $stI->execute($insertPayload + [':url' => $candidate]);
        $slug = $candidate;
        break;
      } catch (PDOException $ex) {
        if ((string)$ex->getCode() === '23000') continue;
        throw $ex;
      }
    }

    if ($slug === '') throw new RuntimeException('URL üretilemedi.');

    $ilanId = (int)db()->lastInsertId();

    $stG = db()->prepare("INSERT INTO ilan_gorselleri (ilan_id, gorsel_url, sirano) VALUES (:ilan_id, :url, :sirano)");
    $sir = 1;
    foreach ($galIds as $gid) {
      $stG->execute([':ilan_id' => $ilanId, ':url' => $map[$gid], ':sirano' => $sir]);
      $sir++;
    }

    if ($val['mulktipleri']) {
      $stP = db()->prepare("INSERT INTO ilan_mulktipleri (ilan_id, mulktipi_id) VALUES (:ilan_id, :pid)");
      foreach ($val['mulktipleri'] as $pid) $stP->execute([':ilan_id' => $ilanId, ':pid' => $pid]);
    }

    if ($val['oda_ozellikleri']) {
      $stP = db()->prepare("INSERT INTO ilan_oda_ozellikleri (ilan_id, oda_ozellik_id) VALUES (:ilan_id, :pid)");
      foreach ($val['oda_ozellikleri'] as $pid) $stP->execute([':ilan_id' => $ilanId, ':pid' => $pid]);
    }

    if ($val['tesis_ozellikleri']) {
      $stP = db()->prepare("INSERT INTO ilan_tesis_ozellikleri (ilan_id, tesis_ozellik_id) VALUES (:ilan_id, :pid)");
      foreach ($val['tesis_ozellikleri'] as $pid) $stP->execute([':ilan_id' => $ilanId, ':pid' => $pid]);
    }

    $stR = db()->prepare("INSERT INTO ilan_rezervasyon_linkleri (ilan_id, platform_id, url, sirano) VALUES (:ilan_id, :pid, :url, :sirano)");
    $sir = 0;
    foreach ($rezClean as $pid => $url) {
      $stR->execute([':ilan_id' => $ilanId, ':pid' => $pid, ':url' => $url, ':sirano' => $sir]);
      $sir++;
    }

    // ilan ekleme TAMAMEN başarılı olduktan sonra kredi harcama log'u oluştur
    if ($krediDusuldu) {
      $stLog = db()->prepare("
        INSERT INTO kredi_harcama_log (tesis_id, ilan_id, harcanan_kredi, aciklama)
        VALUES (:t, :i, :k, 'İlan Ekleme')
      ");
      $stLog->execute([
        ':t' => $tesisId,
        ':i' => $ilanId,
        ':k' => ILAN_KREDI_MALIYETI,
      ]);
    }

    db()->commit();

    if ($saveAsDraft) {
      header('Location: kredi-yonetimi.php?taslak=1&ilan_id=' . $ilanId);
      exit;
    }

    header('Location: ilanlarim.php?f=incelemede&ok=1');
    exit;
  } catch (Throwable $e) {
    db()->rollBack();
    error_log('ilan-ekle hata: ' . $e->getMessage());
    $error = 'Kaydetme başarısız.';
  }
}

}

$csrf = csrf_token();
$creditNow = (int)($u['kredi_puani'] ?? 0);

$schemaJs = json_encode($formSchema, JSON_UNESCAPED_UNICODE | JSON_HEX_TAG | JSON_HEX_AMP | JSON_HEX_APOS | JSON_HEX_QUOT);
?>
<!doctype html>
<html lang="tr">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <title>Yeni İlan Ekle</title>

  <script>
    window.BS_ILAN = { credit: <?= (int)$creditNow ?>, needed: <?= (int)ILAN_KREDI_MALIYETI ?>, krediUrl: 'https://www.bungalowsepeti.com/panel/kredi-yonetimi.php' };
    window.BS_UPLOAD = { csrf: '<?= e($csrf) ?>', listEndpoint: 'ilan-ekle.php', uploadEndpoint: 'ortam-kutuphanesi.php', maxGallery: <?= (int)MAX_GALERI ?> };
    window.BS_RULES = <?= json_encode($formSchema['rules'], JSON_UNESCAPED_UNICODE | JSON_HEX_TAG | JSON_HEX_AMP | JSON_HEX_APOS | JSON_HEX_QUOT) ?>;
    window.BS_FORM_SCHEMA = <?= $schemaJs ?>;
  </script>



</head>
<body class="panel-body ilan-page">
<div class="panel-shell">
  <?php include __DIR__ . '/sidebar.php'; ?>
  <div class="panel-main">
    <header class="topbar">
      <div class="topbar-title">
        <div class="topbar-h1">Yeni İlan Ekle</div>
        <div class="topbar-sub muted">İlan ekleme: <?= (int)ILAN_KREDI_MALIYETI ?> kredi</div>
      </div>
      <div class="topbar-right">
        <a class="btn btn-ghost btn-sm" href="ilanlarim.php">← İlanlarım</a>
      </div>
    </header>

    <main class="content">
      <?php if ($error): ?>
        <script>
          document.addEventListener('DOMContentLoaded', () => {
            const msg = <?= json_encode($error, JSON_UNESCAPED_UNICODE | JSON_HEX_TAG | JSON_HEX_AMP | JSON_HEX_APOS | JSON_HEX_QUOT) ?>;
            window.BS?.toast?.(msg, 'error', 12200);
          });
        </script>
      <?php endif; ?>

      <form method="post" class="ilan-form" id="ilanForm" novalidate>
        <input type="hidden" name="csrf_token" value="<?= e($csrf) ?>">
        <input type="hidden" name="save_as_draft" id="save_as_draft" value="0">
        <input type="hidden" name="sozlesme_onay" id="sozlesme_onay" value="">

        <section class="panel-section ilan-card js-req-card" data-req="category">
          <div class="form-legend">İlan Kategorisi</div>
          <div class="cat-row">
            <?php foreach ($kats as $k): ?>
              <?php $kid = (string)($k['id'] ?? ''); ?>
              <label class="cat-pill">
                <input class="cat-input" type="radio" name="ilan_turu_id" value="<?= e($kid) ?>" <?= $val['ilan_turu_id']===$kid?'checked':'' ?> required>
                <span class="cat-text"><?= e((string)($k['isim'] ?? '')) ?></span>
              </label>
            <?php endforeach; ?>
          </div>
        </section>

        <section class="panel-section ilan-card js-req-card" data-req="features">
          <div class="form-legend">İlan Özellikleri</div>
          <div class="ilan-grid-1">
            <div class="field">
              <input class="input" name="oda_ismi" value="<?= e($val['oda_ismi']) ?>" placeholder="Oda İsmi" required>
            </div>
          </div>

          <div class="ilan-grid-2" style="margin-top:12px;">
            <div class="field">
              <input class="input" name="fiyat" value="<?= e($val['fiyat']) ?>" inputmode="numeric" placeholder="1 Kişi 1 Gece Fiyatı" required>
            </div>
            <div class="field">
              <input class="input" name="indirimli_fiyat" value="<?= e($val['indirimli_fiyat']) ?>" inputmode="numeric" placeholder="İndirimli Fiyat">
            </div>
          </div>

          <div class="ilan-grid-6" style="margin-top:12px;">
            <div class="field">
              <select class="input" name="max_yetiskin" required>
                <option value="" <?= $val['max_yetiskin']===''?'selected':'' ?> disabled hidden>Maks. Yetişkin</option>
                <?php for($i=1;$i<=10;$i++): ?>
                  <option value="<?= $i ?>" <?= $val['max_yetiskin']===(string)$i?'selected':'' ?>><?= $i ?> Yetişkin</option>
                <?php endfor; ?>
              </select>
            </div>
            <div class="field">
              <select class="input" name="max_cocuk" required>
                <option value="" <?= $val['max_cocuk']===''?'selected':'' ?> disabled hidden>Maks. Çocuk</option>
                <?php for($i=0;$i<=10;$i++): ?>
                  <option value="<?= $i ?>" <?= $val['max_cocuk']===(string)$i?'selected':'' ?>><?= $i ?> Çocuk</option>
                <?php endfor; ?>
              </select>
            </div>
            <div class="field">
              <select class="input" name="yatak_odasi" required>
                <option value="" <?= $val['yatak_odasi']===''?'selected':'' ?> disabled hidden>Yatak Odası</option>
                <?php for($i=0;$i<=10;$i++): ?>
                  <option value="<?= $i ?>" <?= $val['yatak_odasi']===(string)$i?'selected':'' ?>><?= $i ?> Yatak Odası</option>
                <?php endfor; ?>
              </select>
            </div>
            <div class="field">
              <select class="input" name="yatak_sayisi" required>
                <option value="" <?= $val['yatak_sayisi']===''?'selected':'' ?> disabled hidden>Yatak Sayısı</option>
                <?php for($i=1;$i<=20;$i++): ?>
                  <option value="<?= $i ?>" <?= $val['yatak_sayisi']===(string)$i?'selected':'' ?>><?= $i ?> Yatak</option>
                <?php endfor; ?>
              </select>
            </div>
            <div class="field">
              <select class="input" name="banyo" required>
                <option value="" <?= $val['banyo']===''?'selected':'' ?> disabled hidden>Banyo</option>
                <?php for($i=0;$i<=10;$i++): ?>
                  <option value="<?= $i ?>" <?= $val['banyo']===(string)$i?'selected':'' ?>><?= $i ?> Banyo</option>
                <?php endfor; ?>
              </select>
            </div>
            <div class="field">
              <select class="input" name="havuz" required>
                <option value="" <?= $val['havuz']===''?'selected':'' ?> disabled hidden>Havuz</option>
                <option value="Yok" <?= $val['havuz']==='Yok'?'selected':'' ?>>Havuz Yok</option>
                <option value="Var" <?= $val['havuz']==='Var'?'selected':'' ?>>Havuz Var</option>
              </select>
            </div>
          </div>

          <div class="ilan-grid-3" style="margin-top:12px;">
            <div class="field">
              <input class="input" name="min_konaklama" value="<?= e($val['min_konaklama']) ?>" inputmode="numeric" placeholder="Minimum Konaklama (Gece)" required>
            </div>
            <div class="field">
              <select class="input" name="evlilik_sarti" required>
                <option value="" <?= $val['evlilik_sarti']===''?'selected':'' ?> disabled hidden>Evlilik Şartı</option>
                <option value="Yok" <?= $val['evlilik_sarti']==='Yok'?'selected':'' ?>>Yok</option>
                <option value="Var" <?= $val['evlilik_sarti']==='Var'?'selected':'' ?>>Var</option>
              </select>
            </div>
            <div class="field">
              <input class="input" name="metrekare" value="<?= e($val['metrekare']) ?>" inputmode="numeric" placeholder="m² Bilgisi" required>
            </div>
          </div>

          <div class="ilan-grid-1" style="margin-top:12px;">
            <div class="field">
              <textarea class="input" name="aciklama" rows="5" style="resize:vertical;" placeholder="İlan Açıklaması" required><?= e($val['aciklama']) ?></textarea>
            </div>
          </div>
        </section>

        <section class="panel-section ilan-card js-req-card" data-req="mulks">
          <div class="form-legend">Mülk Tipleri</div>
          <div class="pill-grid">
            <?php foreach ($mulks as $m): ?>
              <?php $mid = safe_id($m['id'] ?? null); ?>
              <label class="pill">
                <input type="checkbox" name="mulktipleri[]" value="<?= $mid ?>" <?= in_array($mid, $val['mulktipleri'], true)?'checked':'' ?>>
                <span><?= e((string)($m['isim'] ?? '')) ?></span>
              </label>
            <?php endforeach; ?>
          </div>
        </section>

        <section class="panel-section ilan-card js-req-card" data-req="tesis">
          <div class="form-legend">Tesis Özellikleri</div>
          <div class="pill-grid">
            <?php foreach ($tesiss as $m): ?>
              <?php $tid = safe_id($m['id'] ?? null); ?>
              <label class="pill">
                <input type="checkbox" name="tesis_ozellikleri[]" value="<?= $tid ?>" <?= in_array($tid, $val['tesis_ozellikleri'], true)?'checked':'' ?>>
                <span><?= e((string)($m['isim'] ?? '')) ?></span>
              </label>
            <?php endforeach; ?>
          </div>
        </section>

        <section class="panel-section ilan-card js-req-card" data-req="oda">
          <div class="form-legend">Oda Özellikleri</div>
          <div class="pill-grid">
            <?php foreach ($odas as $m): ?>
              <?php $oid = safe_id($m['id'] ?? null); ?>
              <label class="pill">
                <input type="checkbox" name="oda_ozellikleri[]" value="<?= $oid ?>" <?= in_array($oid, $val['oda_ozellikleri'], true)?'checked':'' ?>>
                <span><?= e((string)($m['isim'] ?? '')) ?></span>
              </label>
            <?php endforeach; ?>
          </div>
        </section>

        <section class="panel-section ilan-card js-req-card" data-req="media">
          <div class="form-legend">İlan Görselleri</div>
          <input type="hidden" name="cover_media_id" id="cover_media_id" value="<?= e($val['cover_media_id']) ?>">
          <input type="hidden" name="gallery_media_ids" id="gallery_media_ids" value="<?= e($val['gallery_media_ids']) ?>">

          <div class="media-picks">
            <div class="media-pickbox">
              <button class="btn btn-primary btn-sm" type="button" id="btnPickCover">Kapak Görseli Seç</button>
            </div>
            <div class="media-pickbox">
              <button class="btn btn-primary btn-sm" type="button" id="btnPickGallery">Galeri Görseli Seç</button>
            </div>
          </div>

          <div class="media-preview">
            <div class="media-preview-col">
              <div class="picked-cover-box" id="pickedCover"></div>
            </div>
            <div class="media-preview-col">
              <div class="picked-grid" id="pickedGallery"></div>
            </div>
          </div>
        </section>

        <section class="panel-section ilan-card js-req-card" data-req="distance">
          <div class="form-legend">Mesafe Haritası</div>
          <div class="ilan-grid-1">
            <div class="field float-field">
              <select class="input" name="havaalani_ismi" required>
                <option value="" <?= $val['havaalani_ismi']===''?'selected':'' ?> disabled hidden>En Yakın Havalimanı İsmi</option>
                <?php foreach ($havaalanlari as $ha): ?>
                <option value="<?= e($ha) ?>" <?= $val['havaalani_ismi']===$ha?'selected':'' ?>><?= e($ha) ?></option>
                <?php endforeach; ?>
              </select>
            </div>
          </div>

          <div class="ilan-grid-6" style="margin-top:12px;">
            <div class="field float-field">
              <div class="input-row">
                <input class="input" placeholder="Havalimanı Mesafe" name="havaalani_mesafe" value="<?= e($val['havaalani_mesafe']) ?>" inputmode="numeric" pattern="\d*" autocomplete="off" required>
                <select class="input input-unit" name="hava_mesafe_birim">
                  <option value="km" <?= $val['hava_mesafe_birim']==='km'?'selected':'' ?>>km</option>
                  <option value="m" <?= $val['hava_mesafe_birim']==='m'?'selected':'' ?>>m</option>
                </select>
              </div>
            </div>

            <div class="field float-field">
              <div class="input-row">
                <input class="input" placeholder="Plaj Mesafe" name="plaj_mesafe" value="<?= e($val['plaj_mesafe']) ?>" inputmode="numeric" pattern="\d*" autocomplete="off" required>
                <select class="input input-unit" name="plaj_mesafe_birim">
                  <option value="km" <?= $val['plaj_mesafe_birim']==='km'?'selected':'' ?>>km</option>
                  <option value="m" <?= $val['plaj_mesafe_birim']==='m'?'selected':'' ?>>m</option>
                </select>
              </div>
            </div>

            <div class="field float-field">
              <div class="input-row">
                <input class="input" placeholder="Otogar Mesafe" name="otogar_mesafe" value="<?= e($val['otogar_mesafe']) ?>" inputmode="numeric" pattern="\d*" autocomplete="off" required>
                <select class="input input-unit" name="otogar_birim">
                  <option value="km" <?= $val['otogar_birim']==='km'?'selected':'' ?>>km</option>
                  <option value="m" <?= $val['otogar_birim']==='m'?'selected':'' ?>>m</option>
                </select>
              </div>
            </div>

            <div class="field float-field">
              <div class="input-row">
                <input class="input" placeholder="Market Mesafe" name="market_mesafe" value="<?= e($val['market_mesafe']) ?>" inputmode="numeric" pattern="\d*" autocomplete="off" required>
                <select class="input input-unit" name="market_mesafe_birim">
                  <option value="km" <?= $val['market_mesafe_birim']==='km'?'selected':'' ?>>km</option>
                  <option value="m" <?= $val['market_mesafe_birim']==='m'?'selected':'' ?>>m</option>
                </select>
              </div>
            </div>

            <div class="field float-field">
              <div class="input-row">
                <input class="input" placeholder="Şehir Merkezi Mesafe" name="sehir_merkezi_mesafe" value="<?= e($val['sehir_merkezi_mesafe']) ?>" inputmode="numeric" pattern="\d*" autocomplete="off" required>
                <select class="input input-unit" name="sehir_mesafe_birim">
                  <option value="km" <?= $val['sehir_mesafe_birim']==='km'?'selected':'' ?>>km</option>
                  <option value="m" <?= $val['sehir_mesafe_birim']==='m'?'selected':'' ?>>m</option>
                </select>
              </div>
            </div>

            <div class="field float-field">
              <div class="input-row">
                <input class="input" placeholder="Restaurant Mesafe" name="restaurant_mesafe" value="<?= e($val['restaurant_mesafe']) ?>" inputmode="numeric" pattern="\d*" autocomplete="off" required>
                <select class="input input-unit" name="rest_mesafe_birim">
                  <option value="km" <?= $val['rest_mesafe_birim']==='km'?'selected':'' ?>>km</option>
                  <option value="m" <?= $val['rest_mesafe_birim']==='m'?'selected':'' ?>>m</option>
                </select>
              </div>
            </div>
          </div>
        </section>

        <section class="panel-section ilan-card js-req-card" data-req="rez">
          <div class="form-legend">Rezervasyon Linkleri</div>
          <div class="ilan-grid-4">
            <?php foreach ($plats as $p): $pid = safe_id($p['id'] ?? null); ?>
              <div class="field">
                <input class="input" name="rez[<?= $pid ?>]" value="<?= e((string)($val['rez'][$pid] ?? '')) ?>" placeholder="<?= e((string)($p['isim'] ?? '')) ?>" <?= ($pid === 13) ? 'inputmode="numeric" pattern="[0-9]*" maxlength="11" autocomplete="off"' : '' ?>>
              </div>
            <?php endforeach; ?>
          </div>

          <div class="form-actions">
            <button class="btn btn-primary" type="submit">İlan İncelemeye Gönder</button>
          </div>
        </section>
      </form>
    </main>
  </div>
</div>
</body>
</html>
