<?php
// /home/bungalowsepeti/public_html/panel/ilanlarim.php
// NOT: ilanlar.yayin_durumu enum: taslak, incelemede, yayinda, pasif, arsiv, reddedilen

require_once __DIR__ . '/init.php';

require_login();
auth_refresh_user();

$u = current_user();
$isAdmin = (($u['rol'] ?? '') === 'admin');
$tesisId = (int)($u['id'] ?? 0);

function json_out(array $data, int $code = 200): never {
  http_response_code($code);
  header('Content-Type: application/json; charset=utf-8');
  echo json_encode($data, JSON_UNESCAPED_UNICODE);
  exit;
}

function money_try(?int $n): string {
  $n = (int)($n ?? 0);
  return '₺' . number_format($n, 0, ',', '.');
}

function cover_url(?string $v): string {
  $v = trim((string)$v);
  if ($v === '') return '';
  $base = rtrim(BASE_URL, '/');
  $baseHost = parse_url($base, PHP_URL_HOST);

  if (preg_match('~^https?://~i', $v)) {
    $h = parse_url($v, PHP_URL_HOST);
    if ($baseHost && $h && strcasecmp($h, $baseHost) === 0) return $v;
    return '';
  }

  if (strpos($v, 'uploads/') === 0 || strpos($v, '/uploads/') === 0) return $base . '/' . ltrim($v, '/');
  return $base . '/uploads/' . ltrim($v, '/');
}

function kategori_path($ilan_turu_id): string {
  $map = ['1' => 'bungalov', '2' => 'villa', '3' => 'karavan', '4' => 'tiny-house'];
  return $map[(string)$ilan_turu_id] ?? 'kategori';
}

function kategori_label($ilan_turu_id): string {
  $map = ['1' => 'Bungalov', '2' => 'Villa', '3' => 'Karavan', '4' => 'Tiny House'];
  $k = (string)$ilan_turu_id;
  return $map[$k] ?? ('Kategori ' . $k);
}

function norm_status(string $s): string {
  $v = strtolower(trim($s));
  $v = str_replace(
    ['yayında','onaylı','arşiv','reddedildi','süresi bitti','suresi bitti','süresi-bitti','suresi_bitti'],
    ['yayinda','onayli','arsiv','reddedilen','suresi-bitti','suresi-bitti','suresi-bitti','suresi-bitti'],
    $v
  );
  return $v;
}


function durum_label(string $yayin_durumu): array {
  $v = norm_status($yayin_durumu);

  $textMap = [
    'taslak' => 'Taslak',
    'incelemede' => 'İncelemede',
    'yayinda' => 'Yayında',
    'pasif' => 'Pasif',
    'arsiv' => 'Arşiv',
    'reddedilen' => 'Reddedildi',
    'onayli' => 'İncelemede',
    'yayinda-degil' => 'İncelemede',
    'reddedildi' => 'Reddedildi',
    'suresi-bitti' => 'Süresi Bitti',
  ];

  $toneMap = [
    'taslak' => 'warning',
    'incelemede' => 'info',
    'yayinda' => 'success',
    'pasif' => 'warning',
    'arsiv' => 'archive',
    'reddedilen' => 'danger',
    'onayli' => 'info',
    'yayinda-degil' => 'info',
    'reddedildi' => 'danger',
    'suresi-bitti' => 'warning',
  ];

  if (isset($textMap[$v])) return [$textMap[$v], $toneMap[$v] ?? 'info'];
  return ['İncelemede', 'info'];
}

function status_to_key(string $statusNorm): string {
  $s = norm_status($statusNorm);
  if ($s === 'yayinda') return 'aktif';
  if ($s === 'pasif') return 'pasif';
  if ($s === 'taslak') return 'taslak';
  if ($s === 'arsiv') return 'arsiv';
  if ($s === 'reddedilen') return 'reddedilen';
  if ($s === 'suresi-bitti') return 'suresi-bitti';
  return 'incelemede';
}

function status_code(string $s): int {
  $v = norm_status($s);
  $smap = ['taslak'=>1,'incelemede'=>2,'yayinda'=>3,'pasif'=>4,'arsiv'=>5,'reddedilen'=>6,'suresi-bitti'=>7];

  return (int)($map[$v] ?? 0);
}

function code_to_status(?int $c): string {
  $map = [1=>'taslak',2=>'incelemede',3=>'yayinda',4=>'pasif',5=>'arsiv',6=>'reddedilen',7=>'suresi-bitti'];

  $c = (int)($c ?? 0);
  return (string)($map[$c] ?? '');
}

function code_to_label(?int $c): string {
  $s = code_to_status($c);
  if ($s === '') return '-';
  return (durum_label($s)[0] ?? '-');
}

function badge_style(string $tone): string {
  $tone = strtolower(trim($tone));
  if ($tone === 'success') return 'background:rgba(47,227,159,.14);border:1px solid rgba(47,227,159,.40);color:#0b1635;';
  if ($tone === 'warning') return 'background:rgba(255,184,0,.16);border:1px solid rgba(255,184,0,.42);color:#0b1635;';
  if ($tone === 'danger') return 'background:rgba(255,77,109,.14);border:1px solid rgba(255,77,109,.40);color:#0b1635;';
  if ($tone === 'archive') return 'background:rgba(158,63,255,.14);border:1px solid rgba(158,63,255,.40);color:#0b1635;';
  return 'background:rgba(0,194,255,.14);border:1px solid rgba(0,194,255,.40);color:#0b1635;';
}

function btn_style(string $kind): string {
  $kind = strtolower(trim($kind));
  if ($kind === 'success') return 'border-color:rgba(47,227,159,.40);background:rgba(47,227,159,.14);color:#0b1635;';
  if ($kind === 'warning') return 'border-color:rgba(255,184,0,.40);background:rgba(255,184,0,.14);color:#0b1635;';
  if ($kind === 'archive') return 'border-color:rgba(158,63,255,.35);background:rgba(158,63,255,.12);color:#0b1635;';
  return '';
}

/* ---------------------------
   POST: ilan silme kapalı / durum değiştir / admin modal / reddetme geçmişi
----------------------------*/
if (is_post()) {
  if (!csrf_validate($_POST['csrf_token'] ?? null)) json_out(['ok' => false, 'message' => 'Güvenlik doğrulaması başarısız.'], 403);

  $action = (string)($_POST['action'] ?? '');

  if ($action === 'ajax_delete_ilan') {
    json_out(['ok' => false, 'message' => 'İlan silme kapatıldı.'], 403);
  }

  if ($action === 'ajax_reject_history') {
    $ilanId = (int)($_POST['ilan_id'] ?? 0);
    if ($ilanId <= 0) json_out(['ok' => false, 'message' => 'Geçersiz istek.'], 400);

    $st = db()->prepare("SELECT id, tesis_id FROM ilanlar WHERE id = :id LIMIT 1");
    $st->execute([':id' => $ilanId]);
    $row = $st->fetch();
    if (!$row) json_out(['ok' => false, 'message' => 'İlan bulunamadı.'], 404);

    if (!$isAdmin && (int)$row['tesis_id'] !== $tesisId) json_out(['ok' => false, 'message' => 'Bu ilana işlem yapamazsın.'], 403);

    $rejCode = status_code('reddedilen');

    $lg = db()->prepare("
      SELECT id, eski_durum, yeni_durum, tarih, sebep
      FROM ilan_durum_log
      WHERE ilan_id = :ilan_id AND yeni_durum = :rej
      ORDER BY id DESC
      LIMIT 100
    ");
    $lg->execute([':ilan_id' => $ilanId, ':rej' => $rejCode]);
    $logs = $lg->fetchAll() ?: [];

    $hist = [];
    foreach ($logs as $l) {
      $hist[] = [
        'tarih' => (string)($l['tarih'] ?? ''),
        'eski' => code_to_label($l['eski_durum'] ?? null),
        'yeni' => code_to_label($l['yeni_durum'] ?? null),
        'sebep' => (string)($l['sebep'] ?? ''),
      ];
    }

    json_out(['ok' => true, 'ilan_id' => $ilanId, 'history' => $hist]);
  }

  if ($action === 'ajax_admin_modal') {
    if (!$isAdmin) json_out(['ok' => false, 'message' => 'Yetkisiz.'], 403);

    $ilanId = (int)($_POST['ilan_id'] ?? 0);
    if ($ilanId <= 0) json_out(['ok' => false, 'message' => 'Geçersiz istek.'], 400);

    $st = db()->prepare("
      SELECT i.id, i.tesis_id, i.yayin_durumu, i.reddetme_sebep,
             t.tesis_ismi, t.email, t.tesis_il, t.tesis_ilce
      FROM ilanlar i
      LEFT JOIN tesisler t ON t.id = i.tesis_id
      WHERE i.id = :id
      LIMIT 1
    ");
    $st->execute([':id' => $ilanId]);
    $row = $st->fetch();
    if (!$row) json_out(['ok' => false, 'message' => 'İlan bulunamadı.'], 404);

    $lg = db()->prepare("
      SELECT id, eski_durum, yeni_durum, tarih, sebep
      FROM ilan_durum_log
      WHERE ilan_id = :ilan_id
      ORDER BY id DESC
      LIMIT 50
    ");
    $lg->execute([':ilan_id' => $ilanId]);
    $logs = $lg->fetchAll() ?: [];

    $hist = [];
    foreach ($logs as $l) {
      $hist[] = [
        'tarih' => (string)($l['tarih'] ?? ''),
        'eski' => code_to_label($l['eski_durum'] ?? null),
        'yeni' => code_to_label($l['yeni_durum'] ?? null),
        'sebep' => (string)($l['sebep'] ?? ''),
      ];
    }

    json_out([
      'ok' => true,
      'ilan' => [
        'id' => (int)$row['id'],
        'tesis_id' => (int)$row['tesis_id'],
        'yayin_durumu' => norm_status((string)($row['yayin_durumu'] ?? '')),
        'reddetme_sebep' => (string)($row['reddetme_sebep'] ?? ''),
      ],
      'tesis' => [
        'tesis_ismi' => (string)($row['tesis_ismi'] ?? ''),
        'email' => (string)($row['email'] ?? ''),
        'il' => (string)($row['tesis_il'] ?? ''),
        'ilce' => (string)($row['tesis_ilce'] ?? ''),
      ],
      'history' => $hist,
    ]);
  }

  if ($action === 'ajax_admin_set_status') {
    if (!$isAdmin) json_out(['ok' => false, 'message' => 'Yetkisiz.'], 403);

    $ilanId = (int)($_POST['ilan_id'] ?? 0);
    $newStatus = norm_status((string)($_POST['new_status'] ?? ''));
    $sebep = trim((string)($_POST['sebep'] ?? ''));

    $allowed = ['taslak','incelemede','yayinda','pasif','arsiv','reddedilen','suresi-bitti'];

    if ($ilanId <= 0) json_out(['ok' => false, 'message' => 'Geçersiz istek.'], 400);
    if (!in_array($newStatus, $allowed, true)) json_out(['ok' => false, 'message' => 'Geçersiz durum.'], 400);
    if ($newStatus === 'reddedilen' && $sebep === '') json_out(['ok' => false, 'message' => 'Reddetme sebebi zorunludur.'], 400);

    $st = db()->prepare("SELECT id, yayin_durumu FROM ilanlar WHERE id = :id LIMIT 1");
    $st->execute([':id' => $ilanId]);
    $row = $st->fetch();
    if (!$row) json_out(['ok' => false, 'message' => 'İlan bulunamadı.'], 404);

    $oldStatus = norm_status((string)($row['yayin_durumu'] ?? ''));
    $oldCode = status_code($oldStatus);
    $newCode = status_code($newStatus);

    try {
      db()->beginTransaction();

      $up = db()->prepare("UPDATE ilanlar SET yayin_durumu = :s, reddetme_sebep = :rs WHERE id = :id LIMIT 1");
      $up->execute([
        ':s' => $newStatus,
        ':rs' => ($newStatus === 'reddedilen') ? $sebep : null,
        ':id' => $ilanId
      ]);

      $ins = db()->prepare("INSERT INTO ilan_durum_log (ilan_id, eski_durum, yeni_durum, sebep) VALUES (:ilan_id, :eski, :yeni, :sebep)");
      $ins->execute([
        ':ilan_id' => $ilanId,
        ':eski' => $oldCode ?: null,
        ':yeni' => $newCode ?: null,
        ':sebep' => ($newStatus === 'reddedilen') ? $sebep : null,
      ]);

      db()->commit();
    } catch (Throwable $e) {
      if (db()->inTransaction()) db()->rollBack();
      json_out(['ok' => false, 'message' => 'İşlem başarısız.'], 500);
    }

    [$txt, $tone] = durum_label($newStatus);
    $key = status_to_key($newStatus);

    json_out([
      'ok' => true,
      'ilan_id' => $ilanId,
      'new_status' => $newStatus,
      'label' => $txt,
      'tone' => $tone,
      'badge_style' => badge_style($tone),
      'key' => $key,
    ]);
  }

  if ($action === 'ajax_set_pasif' || $action === 'ajax_archive_ilan' || $action === 'ajax_set_aktif') {
    $ilanId = (int)($_POST['ilan_id'] ?? 0);
    if ($ilanId <= 0) json_out(['ok' => false, 'message' => 'Geçersiz istek.'], 400);

    $st = db()->prepare("SELECT id, tesis_id, yayin_durumu FROM ilanlar WHERE id = :id LIMIT 1");
    $st->execute([':id' => $ilanId]);
    $row = $st->fetch();
    if (!$row) json_out(['ok' => false, 'message' => 'İlan bulunamadı.'], 404);

    if (!$isAdmin && (int)$row['tesis_id'] !== $tesisId) json_out(['ok' => false, 'message' => 'Bu ilana işlem yapamazsın.'], 403);

    $cur = norm_status((string)($row['yayin_durumu'] ?? ''));

    if ($action === 'ajax_set_pasif') {
      if ($cur !== 'yayinda') json_out(['ok' => false, 'message' => 'Sadece Aktif ilanlar Pasif yapılabilir.'], 400);
      $newStatus = 'pasif';
    } elseif ($action === 'ajax_archive_ilan') {
      if ($cur !== 'pasif') json_out(['ok' => false, 'message' => 'Sadece Pasif ilanlar Arşivlenebilir.'], 400);
      $newStatus = 'arsiv';
    } else {
      if ($cur !== 'pasif' && $cur !== 'arsiv') json_out(['ok' => false, 'message' => 'Sadece Pasif / Arşiv ilanlar Aktif yapılabilir.'], 400);
      $newStatus = 'yayinda';
    }

    try {
      $up = db()->prepare("UPDATE ilanlar SET yayin_durumu = :s WHERE id = :id LIMIT 1");
      $up->execute([':s' => $newStatus, ':id' => $ilanId]);
    } catch (Throwable $e) {
      json_out(['ok' => false, 'message' => 'İşlem başarısız.'], 500);
    }

    [$txt, $tone] = durum_label($newStatus);

    json_out([
      'ok' => true,
      'ilan_id' => $ilanId,
      'new_status' => $newStatus,
      'label' => $txt,
      'tone' => $tone,
    ]);
  }

  json_out(['ok' => false, 'message' => 'Geçersiz işlem.'], 400);
}

/* ---------------------------
   LIST: sayfaya girildiğinde tüm ilanlar gelir
   Tab değişiminde SUNUCUYA İSTEK YOK
----------------------------*/
$nowTs = time();
$csrf = csrf_token();

$baseSelect = "
SELECT
  i.id, i.tesis_id, i.oda_ismi, i.ilan_turu_id, i.fiyat, i.indirimli_fiyat, i.kapak_gorsel, i.url, i.yayin_durumu, i.reddetme_sebep,
  oc.bitis_tarih AS one_cikar_bitis
FROM ilanlar i
LEFT JOIN (
  SELECT ilan_id, tesis_id, MAX(bitis_tarih) AS bitis_tarih
  FROM one_cikarilan_ilanlar
  GROUP BY ilan_id, tesis_id
) oc ON oc.ilan_id = i.id AND oc.tesis_id = i.tesis_id
";

if ($isAdmin) {
  $listSql = $baseSelect . " ORDER BY i.id DESC";
  $listSt = db()->prepare($listSql);
  $listSt->execute();
} else {
  $listSql = $baseSelect . " WHERE i.tesis_id = :tesis_id ORDER BY i.id DESC";
  $listSt = db()->prepare($listSql);
  $listSt->execute([':tesis_id' => $tesisId]);
}

$rows = $listSt->fetchAll() ?: [];

$counts = ['aktif'=>0,'pasif'=>0,'taslak'=>0,'incelemede'=>0,'arsiv'=>0,'reddedilen'=>0,'suresi-bitti'=>0];

foreach ($rows as $r) {
  $k = status_to_key((string)($r['yayin_durumu'] ?? ''));
  if (!isset($counts[$k])) $counts[$k] = 0;
  $counts[$k]++;
}
$totalAll = count($rows);
?>
<!doctype html>
<html lang="tr">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <title>İlanlarım</title>
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css" referrerpolicy="no-referrer">
  
</head>
<body class="panel-body">

<div class="panel-shell">
  <?php include __DIR__ . '/sidebar.php'; ?>

  <div class="panel-main">
    <header class="topbar">
      <div class="topbar-title">
        <div class="topbar-h1">İlanlarım</div>
        <div class="topbar-sub muted">Toplam: <span id="ilanTotal"><?= (int)$totalAll ?></span></div>
      </div>

      <div class="topbar-right" style="display:flex;gap:10px;align-items:center;flex-wrap:wrap;justify-content:flex-end;">
        <div id="ilanFilters" style="display:flex;gap:8px;flex-wrap:wrap;justify-content:flex-end;">
          <button class="btn btn-primary btn-sm js-filter" type="button" data-filter="aktif">Aktif <span class="muted js-count" data-k="aktif">(<?= (int)$counts['aktif'] ?>)</span></button>
          <button class="btn btn-ghost btn-sm js-filter" type="button" data-filter="pasif">Pasif <span class="muted js-count" data-k="pasif">(<?= (int)$counts['pasif'] ?>)</span></button>
          <button class="btn btn-ghost btn-sm js-filter" type="button" data-filter="taslak">Taslak <span class="muted js-count" data-k="taslak">(<?= (int)$counts['taslak'] ?>)</span></button>
          <button class="btn btn-ghost btn-sm js-filter" type="button" data-filter="incelemede">İncelemede <span class="muted js-count" data-k="incelemede">(<?= (int)$counts['incelemede'] ?>)</span></button>
          <button class="btn btn-ghost btn-sm js-filter" type="button" data-filter="reddedilen">Reddedilen <span class="muted js-count" data-k="reddedilen">(<?= (int)$counts['reddedilen'] ?>)</span></button>
          <button class="btn btn-ghost btn-sm js-filter" type="button" data-filter="arsiv">Arşiv <span class="muted js-count" data-k="arsiv">(<?= (int)$counts['arsiv'] ?>)</span></button>
          <button class="btn btn-ghost btn-sm js-filter" type="button" data-filter="suresi-bitti">Süresi Bitti <span class="muted js-count" data-k="suresi-bitti">(<?= (int)$counts['suresi-bitti'] ?>)</span></button>

        </div>

        <a class="btn btn-primary btn-sm" href="ilan-ekle.php">+ İlan Ekle</a>
      </div>
    </header>

    <main class="content">
      <section class="panel-section" style="margin-top:0;">
        <div class="section-head">
<div class="section-title" id="ilanSectionTitle">Aktif</div>
        </div>

        <div class="table" role="table" aria-label="İlanlarım">
          <div class="tr th" role="row">
            <div class="td" style="flex:0 0 65px;">İlan No</div>
            <div class="td" style="flex:0 0 90px;">Kapak</div>
            <div class="td">Oda İsmi</div>
            <div class="td" style="flex:0 0 120px;">Kategori</div>
            <div class="td" style="flex:0 0 140px;">Fiyat</div>
            <div class="td" style="flex:0 0 160px;">Öne Çıkar</div>
            <div class="td" style="flex:0 0 140px;">Durumu</div>
            <div class="td td-right" style="flex:0 0 260px;">İşlemler</div>
          </div>

          <div id="ilanRows">
            <?php if (!$rows): ?>
              <div class="tr" role="row">
                <div class="td muted">Henüz ilan yok.</div>
              </div>
            <?php endif; ?>

            <?php foreach ($rows as $r): ?>
              <?php
                $ilanId = (int)$r['id'];
                $cover = cover_url($r['kapak_gorsel'] ?? '');
                $oda = (string)($r['oda_ismi'] ?? '');
                $kat = kategori_label($r['ilan_turu_id'] ?? '');
                $fiyat = (int)($r['fiyat'] ?? 0);
                $ind = (int)($r['indirimli_fiyat'] ?? 0);

                $yDur = norm_status((string)($r['yayin_durumu'] ?? ''));
                $key = status_to_key($yDur);

                [$durumText, $durumTone] = durum_label($yDur);
                $badgeStyle = badge_style($durumTone);

                $oneBitis = (string)($r['one_cikar_bitis'] ?? '');
                $oneActive = false;
                $oneBitisFmt = '';
                if ($oneBitis !== '') {
                  $ts = strtotime($oneBitis);
                  if ($ts && $ts > $nowTs) { $oneActive = true; $oneBitisFmt = date('d.m.Y', $ts); }
                }

                $isAktif = ($yDur === 'yayinda');
                $isPasif = ($yDur === 'pasif');
                $isArsiv = ($yDur === 'arsiv');

                $oneBtnDisabled = (!$isAktif || $oneActive);

                $slug = trim((string)($r['url'] ?? ''));
$katPath = kategori_path($r['ilan_turu_id'] ?? '');
$rowTesisId = (int)($r['tesis_id'] ?? 0);
$prev = $slug ? (rtrim(BASE_URL,'/').'/'.$katPath.'/'.$rowTesisId.'/'.$slug) : '';

              ?>

              <div class="tr" role="row" data-ilan-row="<?= $ilanId ?>" data-key="<?= e($key) ?>">
                <div class="td" style="flex:0 0 65px;"><?= $ilanId ?></div>

                <div class="td" style="flex:0 0 90px;">
                  <?php if ($cover): ?>
                    <img src="<?= e($cover) ?>" alt="" style="width:72px;height:44px;object-fit:cover;border-radius:10px;border:1px solid rgba(11,22,53,.12);">
                  <?php else: ?>
                    <div style="width:72px;height:44px;border-radius:10px;border:1px dashed rgba(11,22,53,.18);display:flex;align-items:center;justify-content:center;color:rgba(11,22,53,.45);font-size:12px;">—</div>
                  <?php endif; ?>
                </div>

                <div class="td" title="<?= e($oda) ?>">
                  <div style="font-weight:700;white-space:nowrap;overflow:hidden;text-overflow:ellipsis;"><?= e($oda ?: '-') ?></div>
                  <?php if ($prev): ?>
                    <a class="muted" href="<?= e($prev) ?>" target="_blank" rel="noopener" style="font-size:12px;white-space:nowrap;overflow:hidden;text-overflow:ellipsis;display:inline-block;">ÖNİZLEME</a>
                  <?php endif; ?>
                </div>

                <div class="td" style="flex:0 0 120px;"><?= e($kat) ?></div>

                <div class="td" style="flex:0 0 140px;">
                  <?php if ($ind > 0 && $ind < $fiyat): ?>
                    <div style="font-weight:800;"><?= e(money_try($ind)) ?></div>
                    <div class="muted" style="font-size:12px;text-decoration:line-through;"><?= e(money_try($fiyat)) ?></div>
                  <?php else: ?>
                    <div style="font-weight:800;"><?= e(money_try($fiyat)) ?></div>
                  <?php endif; ?>
                </div>

                <div class="td" style="flex:0 0 160px;">
                  <?php if ($oneActive): ?>
                    <span class="badge" title="Bitiş: <?= e($oneBitisFmt) ?>"><span class="dot"></span>Aktif</span>
                  <?php else: ?>
                    <a class="btn btn-ghost btn-sm <?= $oneBtnDisabled ? 'is-disabled' : '' ?>"
                       href="<?= $oneBtnDisabled ? 'javascript:void(0)' : 'one-cikar.php?ilan_id='.$ilanId ?>"
                       style="<?= $oneBtnDisabled ? 'pointer-events:none;opacity:.55;' : '' ?>">Öne Çıkar</a>
                  <?php endif; ?>
                </div>

                <div class="td" data-cell="durum" style="flex:0 0 140px;">
                  <div class="ilan-status-wrap" style="display:flex;align-items:center;gap:6px;">
                    <span class="badge js-durum-badge" style="<?= $badgeStyle ?> padding:0px 15px;"><?= e($durumText) ?></span>
                    <?php if ($yDur === 'reddedilen'): ?>
                      <button class="reject-info-btn js-reject-info" type="button" data-ilan-id="<?= $ilanId ?>" aria-label="Reddetme geçmişi"><i class="fa-solid fa-circle-info"></i></button>
                    <?php endif; ?>
                  </div>
                </div>

                <div class="td td-right" style="flex:0 0 260px;display:flex;justify-content:flex-end;gap:8px;align-items:center;flex-wrap:wrap;">
                  <a class="btn btn-ghost btn-sm" href="ilan-duzenle.php?id=<?= $ilanId ?>">Düzenle</a>

                  <?php if ($isAktif): ?>
                    <button class="btn btn-sm js-ilan-pasif" style="<?= btn_style('warning') ?>" type="button" data-ilan-id="<?= $ilanId ?>" data-ilan-name="<?= e($oda ?: ('İlan #'.$ilanId)) ?>">Pasif Yap</button>
                  <?php elseif ($isPasif): ?>
                    <button class="btn btn-sm js-ilan-aktif" style="<?= btn_style('success') ?>" type="button" data-ilan-id="<?= $ilanId ?>" data-ilan-name="<?= e($oda ?: ('İlan #'.$ilanId)) ?>">Aktif Yap</button>
                    <button class="btn btn-danger btn-sm js-ilan-arsiv" type="button" data-ilan-id="<?= $ilanId ?>" data-ilan-name="<?= e($oda ?: ('İlan #'.$ilanId)) ?>">Arşivle</button>
                  <?php elseif ($isArsiv): ?>
                    <button class="btn btn-sm js-ilan-aktif" style="<?= btn_style('success') ?>" type="button" data-ilan-id="<?= $ilanId ?>" data-ilan-name="<?= e($oda ?: ('İlan #'.$ilanId)) ?>">Aktif Yap</button>
                  <?php endif; ?>

                  <?php if ($isAdmin): ?>
                    <button class="btn btn-ghost btn-sm js-admin-gear" type="button" data-ilan-id="<?= $ilanId ?>" aria-label="Yönet"><i class="fa-solid fa-gear"></i></button>
                  <?php endif; ?>
                </div>
              </div>

            <?php endforeach; ?>
          </div>
        </div>
      </section>
    </main>
  </div>
</div>

<script>
window.BS_ILANLAR = { endpoint: "ilanlarim.php", csrf: <?= json_encode($csrf, JSON_UNESCAPED_UNICODE) ?>, isAdmin: <?= $isAdmin ? 'true' : 'false' ?>, btnSuccess: <?= json_encode(btn_style('success'), JSON_UNESCAPED_UNICODE) ?>, btnWarning: <?= json_encode(btn_style('warning'), JSON_UNESCAPED_UNICODE) ?> };
</script>

<script>
document.addEventListener("DOMContentLoaded", () => {
  window.BS = window.BS || {};

  const S = window.BS_ILANLAR || {};
  const endpoint = String(S.endpoint || "ilanlarim.php");
  const csrf = String(S.csrf || "");
  const isAdmin = !!S.isAdmin;
  const STYLE_SUCCESS = String(S.btnSuccess || "");
  const STYLE_WARNING = String(S.btnWarning || "");

  const toast = (m, t = "info", ms = 3200) => {
    if (window.BS && typeof window.BS.toast === "function") {
      window.BS.toast(m, t, ms);
      return;
    }
    // son çare fallback
    try { console.log("[toast]", t, m); } catch {}
    if (t === "error") window.alert(String(m || "İşlem başarısız."));
  };

  const confirmModal = async (opts) => {
    if (window.BS && typeof window.BS.confirmModal === "function") return await window.BS.confirmModal(opts);
    return window.confirm(String(opts?.text || "Emin misiniz?"));
  };

  const qs = (s, r = document) => r.querySelector(s);
  const qsa = (s, r = document) => Array.from(r.querySelectorAll(s));

  const titleEl = qs("#ilanSectionTitle");
  const titleMap = {
    "aktif": "Aktif",
    "pasif": "Pasif",
    "taslak": "Taslak",
    "incelemede": "İncelemede",
    "arsiv": "Arşiv",
    "reddedilen": "Reddedilen",
    "suresi-bitti": "Süresi Bitti"
  };

  const totalEl = qs("#ilanTotal");
  const filterWrap = qs("#ilanFilters");
  const filterBtns = qsa(".js-filter");

  const getRows = () => qsa('#ilanRows .tr[data-ilan-row]');

  const setActiveFilterBtn = (activeKey) => {
    filterBtns.forEach((b) => {
      const f = String(b.dataset.filter || "");
      const active = (f === activeKey);
      b.classList.toggle("btn-primary", active);
      b.classList.toggle("btn-ghost", !active);
    });
  };

  const applyFilter = (filterKey) => {
    const key = String(filterKey || "aktif");
    const rows = getRows();

    rows.forEach((r) => {
      const k = String(r.dataset.key || "incelemede");
      r.style.display = (k === key) ? "" : "none";
    });

    const visible = rows.reduce((acc, r) => acc + (r.style.display === "none" ? 0 : 1), 0);
    if (totalEl) totalEl.textContent = String(visible);

    setActiveFilterBtn(key);
    if (titleEl) titleEl.textContent = titleMap[key] || "Liste";
    try { sessionStorage.setItem("bs_ilan_filter", key); } catch {}

    if (history.replaceState) {
      const u = new URL(window.location.href);
      u.searchParams.set("f", key);
      history.replaceState(null, "", u.toString());
    }
  };

  const post = async (data) => {
    const fd = new FormData();
    fd.append("csrf_token", csrf);
    Object.keys(data || {}).forEach((k) => fd.append(k, String(data[k])));

    const res = await fetch(endpoint, {
      method: "POST",
      body: fd,
      cache: "no-store",
      credentials: "same-origin"
    });

    let json = {};
    try { json = await res.json(); } catch { json = {}; }
    if (!res.ok || !json.ok) throw (json && json.message) ? json.message : "İşlem başarısız.";
    return json;
  };

  const url = new URL(window.location.href);
  const urlFilter = String(url.searchParams.get("f") || "");
  let initialFilter = urlFilter;

  if (!initialFilter) {
    try { initialFilter = String(sessionStorage.getItem("bs_ilan_filter") || ""); } catch {}
  }
  if (!initialFilter) initialFilter = "aktif";

  applyFilter(initialFilter);

  const toastKey = String(url.searchParams.get("toast") || "");
  if (toastKey) {
    const msg =
      toastKey === "pasif" ? "İlan pasif yapıldı." :
      toastKey === "arsiv" ? "İlan arşive alındı." :
      toastKey === "aktif" ? "İlan aktif yapıldı." :
      "";

    if (msg) toast(msg, "success", 2600);

    if (history.replaceState) {
      url.searchParams.delete("toast");
      history.replaceState(null, "", url.toString());
    }
  }

  filterWrap?.addEventListener("click", (e) => {
    const btn = e.target.closest(".js-filter");
    if (!btn) return;
    applyFilter(String(btn.dataset.filter || "aktif"));
  });

  const recalcCountsFromDom = () => {
    const rows = getRows();
    const map = {};
    rows.forEach((r) => {
      const k = String(r.dataset.key || "incelemede");
      map[k] = (map[k] || 0) + 1;
    });
    qsa(".js-count").forEach((el) => {
      const k = String(el.dataset.k || "");
      const n = Number(map[k] || 0);
      el.textContent = "(" + n + ")";
    });
  };

  const renderOpsButtons = (rowEl, statusNorm) => {
    const ops = rowEl?.querySelector(".td-right");
    if (!ops) return;

    const ilanId = String(rowEl.dataset.ilanRow || "");
    const titleCell = rowEl.querySelector('.td[title]');
    const nameEl = titleCell ? titleCell.querySelector('div') : null;
    const ilanName = (nameEl?.textContent || "").trim() || ("İlan #" + ilanId);

    const keepEdit = ops.querySelector('a[href^="ilan-duzenle.php"]');
    const keepGear = ops.querySelector(".js-admin-gear");

    ops.innerHTML = "";
    if (keepEdit) ops.appendChild(keepEdit);

    const mkBtn = (cls, text, style) => {
      const b = document.createElement("button");
      b.type = "button";
      b.className = "btn btn-sm " + cls;
      if (style) b.setAttribute("style", style);
      b.dataset.ilanId = ilanId;
      b.dataset.ilanName = ilanName;
      b.textContent = text;
      return b;
    };

    if (statusNorm === "yayinda") {
      ops.appendChild(mkBtn("js-ilan-pasif", "Pasif Yap", STYLE_WARNING));
    } else if (statusNorm === "pasif") {
      ops.appendChild(mkBtn("js-ilan-aktif", "Aktif Yap", STYLE_SUCCESS));
      ops.appendChild(mkBtn("js-ilan-arsiv btn-danger", "Arşivle", ""));
    } else if (statusNorm === "arsiv") {
      ops.appendChild(mkBtn("js-ilan-aktif", "Aktif Yap", STYLE_SUCCESS));
    }

    if (keepGear) ops.appendChild(keepGear);
  };

  const renderDurumCell = (rowEl, labelText, badgeInlineStyle, isRejected) => {
    const cell = rowEl?.querySelector('.td[data-cell="durum"]');
    if (!cell) return;
    const ilanId = String(rowEl.dataset.ilanRow || "");

    const wrap = document.createElement("div");
    wrap.className = "ilan-status-wrap";
    wrap.setAttribute("style", "display:flex;align-items:center;gap:6px;");

    const badge = document.createElement("span");
    badge.className = "badge js-durum-badge";
    badge.setAttribute("style", String(badgeInlineStyle || "") + " padding:0px 15px;");
    badge.textContent = String(labelText || "");
    wrap.appendChild(badge);

    if (isRejected) {
      const btn = document.createElement("button");
      btn.type = "button";
      btn.className = "reject-info-btn js-reject-info";
      btn.dataset.ilanId = ilanId;
      btn.setAttribute("aria-label", "Reddetme geçmişi");
      btn.innerHTML = '<i class="fa-solid fa-circle-info"></i>';
      wrap.appendChild(btn);
    }

    cell.innerHTML = "";
    cell.appendChild(wrap);
  };

  document.addEventListener("click", async (e) => {
    const pasifBtn = e.target.closest(".js-ilan-pasif");
    const arsivBtn = e.target.closest(".js-ilan-arsiv");
    const aktifBtn = e.target.closest(".js-ilan-aktif");
    if (!pasifBtn && !arsivBtn && !aktifBtn) return;

    const btn = pasifBtn || arsivBtn || aktifBtn;
    const ilanId = String(btn.dataset.ilanId || "");
    const ilanName = String(btn.dataset.ilanName || "").trim() || ("İlan #" + ilanId);
    if (!ilanId) return;

    const mode = pasifBtn ? "pasif" : (arsivBtn ? "arsiv" : "aktif");

    if (mode === "pasif" || mode === "arsiv") {
      const title = (mode === "pasif") ? "Pasife Al" : "Arşivle";
      const text = (mode === "pasif") ? `${ilanName} pasif yapılsın mı?` : `${ilanName} arşive alınsın mı?`;
      const ok = await confirmModal({ title, text, okText: "Evet", cancelText: "Vazgeç" });
      if (!ok) return;
    }

    const action = mode === "pasif" ? "ajax_set_pasif" : (mode === "arsiv" ? "ajax_archive_ilan" : "ajax_set_aktif");

    try {
      await post({ action, ilan_id: ilanId });

      const curUrl = new URL(window.location.href);
      let f = String(curUrl.searchParams.get("f") || "");
      if (!f) { try { f = String(sessionStorage.getItem("bs_ilan_filter") || ""); } catch {} }
      if (!f) f = "aktif";

      curUrl.searchParams.set("f", f);
      curUrl.searchParams.set("toast", mode);
      window.location.href = curUrl.toString();
    } catch (err) {
      toast(String(err || "İşlem başarısız."), "error", 3400);
    }
  });

  const rejectModal = qs("#rejectHistoryModal");
  const rejectList = qs("#rejectHistoryList");

  const rejectOpen = () => {
    if (!rejectModal) return;
    rejectModal.classList.add("is-open");
    rejectModal.setAttribute("aria-hidden", "false");
  };

  const rejectClose = () => {
    if (!rejectModal) return;
    rejectModal.classList.remove("is-open");
    rejectModal.setAttribute("aria-hidden", "true");
  };

  const renderRejectHistory = (items) => {
    if (!rejectList) return;
    const arr = Array.isArray(items) ? items : [];
    if (!arr.length) { rejectList.textContent = "Kayıt yok."; return; }

    rejectList.innerHTML = arr.map((h) => {
      const tarih = String(h?.tarih || "");
      const eski = String(h?.eski || "-");
      const yeni = String(h?.yeni || "-");
      const sebep = String(h?.sebep || "");
      const safeSebep = sebep.replaceAll("&","&amp;").replaceAll("<","&lt;").replaceAll(">","&gt;");
      return `
        <div class="reject-history-item">
          <div class="reject-history-top">
            <span class="reject-history-date">${tarih}</span>
            <span class="reject-history-change">${eski} → ${yeni}</span>
          </div>
          ${sebep ? `<div class="reject-history-reason">Sebep: ${safeSebep}</div>` : ``}
        </div>
      `;
    }).join("");
  };

  rejectModal?.addEventListener("click", (e) => {
    if (e.target === rejectModal) rejectClose();
  });
  qsa(".js-reject-modal-close").forEach((b) => b.addEventListener("click", rejectClose));

  document.addEventListener("click", async (e) => {
    const infoBtn = e.target.closest(".js-reject-info");
    if (!infoBtn) return;

    const ilanId = String(infoBtn.dataset.ilanId || "");
    if (!ilanId) return;

    try {
      const resp = await post({ action: "ajax_reject_history", ilan_id: ilanId });
      renderRejectHistory(resp.history || []);
      rejectOpen();
    } catch (err) {
      toast(String(err || "İşlem başarısız."), "error", 3400);
    }
  });

  if (isAdmin) {
    const adminModal = qs("#adminIlanModal");
    const adminIlanIdEl = qs("#adminIlanId");
    const adminStatusSelect = qs("#adminStatusSelect");
    const adminRejectWrap = qs("#adminRejectWrap");
    const adminRejectReason = qs("#adminRejectReason");
    const adminHistoryList = qs("#adminHistoryList");

    const adminOpen = () => {
      if (!adminModal) return;
      adminModal.classList.add("is-open");
      adminModal.setAttribute("aria-hidden", "false");
    };

    const adminClose = () => {
      if (!adminModal) return;
      adminModal.classList.remove("is-open");
      adminModal.setAttribute("aria-hidden", "true");
    };

    const setRejectVisibility = () => {
      const v = String(adminStatusSelect?.value || "");
      const show = (v === "reddedilen");
      if (adminRejectWrap) adminRejectWrap.style.display = show ? "" : "none";
      if (!show && adminRejectReason) adminRejectReason.value = "";
    };

    const renderHistory = (items) => {
      if (!adminHistoryList) return;
      const arr = Array.isArray(items) ? items : [];
      if (!arr.length) { adminHistoryList.textContent = "Kayıt yok."; return; }

      adminHistoryList.innerHTML = arr.map((h) => {
        const tarih = String(h?.tarih || "");
        const eski = String(h?.eski || "-");
        const yeni = String(h?.yeni || "-");
        const sebep = String(h?.sebep || "");
        const safeSebep = sebep.replaceAll("&","&amp;").replaceAll("<","&lt;").replaceAll(">","&gt;");
        return `
          <div class="admin-modal-history-item">
            <div class="admin-modal-history-top">
              <span class="admin-modal-history-date">${tarih}</span>
              <span class="admin-modal-history-change">${eski} → ${yeni}</span>
            </div>
            ${sebep ? `<div class="admin-modal-history-reason">Sebep: ${safeSebep}</div>` : ``}
          </div>
        `;
      }).join("");
    };

    const fillModal = (payload) => {
      const tesis = payload?.tesis || {};
      const ilan = payload?.ilan || {};

      const name = String(tesis.tesis_ismi || "—");
      const email = String(tesis.email || "—");
      const il = String(tesis.il || "");
      const ilce = String(tesis.ilce || "");
      const ililce = (il || ilce) ? (il + (ilce ? (" / " + ilce) : "")) : "—";

      qsa("[data-field]").forEach((el) => {
        const k = String(el.getAttribute("data-field") || "");
        if (k === "tesis_ismi") el.textContent = name;
        if (k === "email") el.textContent = email;
        if (k === "ililce") el.textContent = ililce;
      });

      if (adminIlanIdEl) adminIlanIdEl.value = String(ilan.id || 0);
      if (adminStatusSelect) adminStatusSelect.value = String(ilan.yayin_durumu || "incelemede");

      if (String(ilan.yayin_durumu || "") === "reddedilen") {
        if (adminRejectReason) adminRejectReason.value = String(ilan.reddetme_sebep || "");
      } else {
        if (adminRejectReason) adminRejectReason.value = "";
      }

      setRejectVisibility();
      renderHistory(payload?.history || []);
    };

    adminModal?.addEventListener("click", (e) => {
      if (e.target === adminModal) adminClose();
    });

    qsa(".js-admin-modal-close").forEach((b) => b.addEventListener("click", adminClose));
    adminStatusSelect?.addEventListener("change", setRejectVisibility);

    document.addEventListener("click", async (e) => {
      const gear = e.target.closest(".js-admin-gear");
      if (!gear) return;

      const ilanId = String(gear.dataset.ilanId || "");
      if (!ilanId) return;

      try {
        const payload = await post({ action: "ajax_admin_modal", ilan_id: ilanId });
        fillModal(payload);
        adminOpen();
      } catch (err) {
        toast(String(err || "İşlem başarısız."), "error", 3400);
      }
    });

    qs(".js-admin-status-save")?.addEventListener("click", async () => {
      const ilanId = String(adminIlanIdEl?.value || "");
      const newStatus = String(adminStatusSelect?.value || "");
      const sebep = String(adminRejectReason?.value || "").trim();

      if (!ilanId) return;

      if (newStatus === "reddedilen" && !sebep) {
        toast("Reddetme sebebi zorunludur.", "error", 3400);
        adminRejectReason?.focus();
        return;
      }

      try {
        const resp = await post({ action: "ajax_admin_set_status", ilan_id: ilanId, new_status: newStatus, sebep });

        const row = qs(`.tr[data-ilan-row="${ilanId}"]`);
        if (row) {
          row.dataset.key = String(resp.key || "incelemede");
          renderDurumCell(row, String(resp.label || ""), String(resp.badge_style || ""), String(resp.new_status || "") === "reddedilen");
          renderOpsButtons(row, String(resp.new_status || ""));
        }

        recalcCountsFromDom();

        const curUrl = new URL(window.location.href);
        let f = String(curUrl.searchParams.get("f") || "");
        if (!f) { try { f = String(sessionStorage.getItem("bs_ilan_filter") || ""); } catch {} }
        if (!f) f = "aktif";
        applyFilter(f);

        toast("Durum güncellendi.", "success", 2400);

        const payload = await post({ action: "ajax_admin_modal", ilan_id: ilanId });
        fillModal(payload);
      } catch (err) {
        toast(String(err || "İşlem başarısız."), "error", 3400);
      }
    });
  }
});
</script>

<?php if ((string)($_GET['ok'] ?? '') === '1'): ?>
<script>
window.addEventListener('load', () => {
  setTimeout(() => {
    window.BS?.toast?.('Tebrikler. İlanınız tekrar incelemeye alındı.', 'success', 4000);

    if (history.replaceState) {
      const u = new URL(window.location.href);
      u.searchParams.delete('ok');
      history.replaceState(null, '', u.toString());
    }
  }, 50);
});
</script>
<?php endif; ?>

<style>
.reject-modal{position:fixed;inset:0;display:none;align-items:center;justify-content:center;background:rgba(0,0,0,.45);z-index:9998;padding:18px;}
.reject-modal.is-open{display:flex;}
.reject-modal-card{width:min(720px,100%);background:#fff;border-radius:14px;box-shadow:0 20px 70px rgba(0,0,0,.25);overflow:hidden;max-height:90vh;display:flex;flex-direction:column;}
.reject-modal-head{display:flex;align-items:center;justify-content:space-between;padding:14px 16px;border-bottom:1px solid rgba(11,22,53,.10);}
.reject-modal-title{font-weight:800;font-size:16px;color:#0b1635;}
.reject-modal-x{border:0;background:transparent;font-size:18px;cursor:pointer;line-height:1;padding:6px 10px;}
.reject-modal-body{padding:14px 16px;overflow:hidden;display:flex;flex-direction:column;gap:10px;}
.reject-history-list{overflow:auto;flex:1;border:1px solid rgba(11,22,53,.10);border-radius:12px;padding:12px;display:flex;flex-direction:column;gap:8px;}
.reject-history-item{border:1px solid rgba(11,22,53,.10);border-radius:10px;padding:10px;}
.reject-history-top{display:flex;gap:10px;justify-content:space-between;flex-wrap:wrap;}
.reject-history-date{font-size:12px;color:rgba(11,22,53,.55);}
.reject-history-change{font-weight:800;color:#0b1635;}
.reject-history-reason{margin-top:6px;font-size:13px;color:rgba(11,22,53,.70);white-space:pre-wrap;}
.reject-info-btn{border:0;background:transparent;padding:0 4px;margin-left:6px;cursor:pointer;color:rgba(11,22,53,.70);}
.reject-info-btn:hover{color:#0b1635;}
</style>

<div id="rejectHistoryModal" class="reject-modal" aria-hidden="true">
  <div class="reject-modal-card" role="dialog" aria-modal="true" aria-labelledby="rejectHistoryTitle">
    <div class="reject-modal-head">
      <div id="rejectHistoryTitle" class="reject-modal-title">Reddetme Geçmişi</div>
      <button type="button" class="reject-modal-x js-reject-modal-close" aria-label="Kapat">✕</button>
    </div>
    <div class="reject-modal-body">
      <div id="rejectHistoryList" class="reject-history-list">—</div>
    </div>
  </div>
</div>

<?php if ($isAdmin): ?>
<style>
.admin-modal{position:fixed;inset:0;display:none;align-items:center;justify-content:center;background:rgba(0,0,0,.45);z-index:9999;padding:18px;}
.admin-modal.is-open{display:flex;}
.admin-modal-card{width:min(920px,100%);background:#fff;border-radius:14px;box-shadow:0 20px 70px rgba(0,0,0,.25);overflow:hidden;max-height:90vh;display:flex;flex-direction:column;}
.admin-modal-head{display:flex;align-items:center;justify-content:space-between;padding:14px 16px;border-bottom:1px solid rgba(11,22,53,.10);}
.admin-modal-title{font-weight:800;font-size:16px;color:#0b1635;}
.admin-modal-x{border:0;background:transparent;font-size:18px;cursor:pointer;line-height:1;padding:6px 10px;}
.admin-modal-body{padding:14px 16px;flex:1;overflow:hidden;display:flex;flex-direction:column;}
.admin-modal-grid{display:grid;grid-template-columns:1fr 1fr;gap:14px;flex:0 0 auto;}
.admin-modal-col{border:1px solid rgba(11,22,53,.10);border-radius:12px;padding:12px;}
.admin-modal-row{display:flex;gap:10px;align-items:flex-start;padding:8px 0;border-bottom:1px dashed rgba(11,22,53,.10);}
.admin-modal-row:last-child{border-bottom:0;}
.admin-modal-label{width:110px;flex:0 0 110px;font-weight:700;color:#0b1635;}
.admin-modal-val{flex:1;color:#0b1635;word-break:break-word;}
.admin-modal-label2{display:block;font-weight:800;color:#0b1635;margin:2px 0 6px;}
.admin-modal-select{width:100%;padding:10px 12px;border-radius:10px;border:1px solid rgba(11,22,53,.18);outline:0;}
.admin-modal-reject{margin-top:10px;}
.admin-modal-textarea{width:100%;padding:10px 12px;border-radius:10px;border:1px solid rgba(11,22,53,.18);outline:0;resize:vertical;min-height:96px;}
.admin-modal-actions{display:flex;gap:8px;justify-content:flex-end;margin-top:12px;flex-wrap:wrap;}
.admin-modal-history{margin-top:14px;border:1px solid rgba(11,22,53,.10);border-radius:12px;padding:12px;flex:1;overflow:hidden;display:flex;flex-direction:column;}
.admin-modal-history-title{font-weight:900;color:#0b1635;margin-bottom:8px;flex:0 0 auto;}
.admin-modal-history-list{display:flex;flex-direction:column;gap:8px;flex:1;overflow:auto;}
.admin-modal-history-item{border:1px solid rgba(11,22,53,.10);border-radius:10px;padding:10px;}
.admin-modal-history-top{display:flex;gap:10px;justify-content:space-between;flex-wrap:wrap;}
.admin-modal-history-date{font-size:12px;color:rgba(11,22,53,.55);}
.admin-modal-history-change{font-weight:800;color:#0b1635;}
.admin-modal-history-reason{margin-top:6px;font-size:13px;color:rgba(11,22,53,.70);white-space:pre-wrap;}
</style>

<div id="adminIlanModal" class="admin-modal" aria-hidden="true">
  <div class="admin-modal-card" role="dialog" aria-modal="true" aria-labelledby="adminIlanModalTitle">
    <div class="admin-modal-head">
      <div id="adminIlanModalTitle" class="admin-modal-title">İlan Yönetimi</div>
      <button type="button" class="admin-modal-x js-admin-modal-close" aria-label="Kapat">✕</button>
    </div>

    <div class="admin-modal-body">
      <div class="admin-modal-grid">
        <div class="admin-modal-col">
          <div class="admin-modal-row">
            <div class="admin-modal-label">Tesis İsmi:</div>
            <div class="admin-modal-val" data-field="tesis_ismi">—</div>
          </div>
          <div class="admin-modal-row">
            <div class="admin-modal-label">E-Posta:</div>
            <div class="admin-modal-val" data-field="email">—</div>
          </div>
          <div class="admin-modal-row">
            <div class="admin-modal-label">İl / İlçe:</div>
            <div class="admin-modal-val" data-field="ililce">—</div>
          </div>
        </div>

        <div class="admin-modal-col">
          <label class="admin-modal-label2" for="adminStatusSelect">Durum Seçin</label>
          <select id="adminStatusSelect" class="admin-modal-select">
            <option value="taslak">Taslak</option>
            <option value="incelemede">İncelemede</option>
            <option value="yayinda">Yayında</option>
            <option value="pasif">Pasif</option>
            <option value="arsiv">Arşiv</option>
            <option value="reddedilen">Reddedildi</option>
            <option value="suresi-bitti">Süresi Bitti</option>
          </select>

          <div id="adminRejectWrap" class="admin-modal-reject" style="display:none;">
            <label class="admin-modal-label2" for="adminRejectReason">Reddetme Sebebi</label>
            <textarea id="adminRejectReason" class="admin-modal-textarea" rows="4" placeholder="Reddetme sebebi zorunludur."></textarea>
          </div>

          <div class="admin-modal-actions">
            <button type="button" class="btn btn-primary btn-sm js-admin-status-save">Kaydet</button>
            <button type="button" class="btn btn-ghost btn-sm js-admin-modal-close">Kapat</button>
          </div>
        </div>
      </div>

      <div class="admin-modal-history">
        <div class="admin-modal-history-title">Durum Geçmişi:</div>
        <div id="adminHistoryList" class="admin-modal-history-list">—</div>
      </div>

      <input type="hidden" id="adminIlanId" value="0">
    </div>
  </div>
</div>
<?php endif; ?>

</body>
</html>
