// /home/bungalowsepeti/public_html/panel/js/sidebar.js
(() => {
  "use strict";

  const STORAGE_KEY = "bs_panel_sidebar_v1";
  const body = document.body;

  const isMobile = () => window.matchMedia("(max-width: 760px)").matches;

  const safeParse = (v) => {
    try { return JSON.parse(v); } catch { return null; }
  };

  const loadState = () => {
    const raw = localStorage.getItem(STORAGE_KEY);
    const parsed = raw ? safeParse(raw) : null;
    return {
      collapsed: !!(parsed && parsed.collapsed),
      mobileOpen: !!(parsed && parsed.mobileOpen),
      groups: (parsed && typeof parsed.groups === "object" && parsed.groups) ? parsed.groups : {}
    };
  };

  const saveState = (state) => {
    try { localStorage.setItem(STORAGE_KEY, JSON.stringify(state)); } catch {}
  };

  const slugify = (s) => (s || "")
    .toString()
    .trim()
    .toLowerCase()
    .replace(/\s+/g, "-")
    .replace(/[^\w\-ığüşöçİĞÜŞÖÇ]+/g, "")
    .replace(/\-+/g, "-");

  const ensureOverlay = () => {
    let overlay = document.querySelector(".sidebar-overlay");
    if (!overlay) {
      overlay = document.createElement("div");
      overlay.className = "sidebar-overlay";
      overlay.setAttribute("aria-hidden", "true");
      document.body.appendChild(overlay);
    }
    return overlay;
  };

  const setSidebarAria = () => {
    const sidebar = document.getElementById("sidebar");
    if (!sidebar) return;

    if (isMobile()) {
      const open = body.classList.contains("is-sidebar-open");
      sidebar.setAttribute("aria-hidden", open ? "false" : "true");
    } else {
      sidebar.setAttribute("aria-hidden", "false");
    }
  };

  const setToggleAria = () => {
    const expandedDesktop = !body.classList.contains("is-sidebar-collapsed");
    document.querySelectorAll(".sidebar-opener").forEach((btn) => btn.setAttribute("aria-expanded", expandedDesktop ? "true" : "false"));
    document.querySelectorAll(".sidebar-close").forEach((btn) => btn.setAttribute("aria-expanded", expandedDesktop ? "true" : "false"));
  };

  const applyLayoutState = (state) => {
    if (isMobile()) {
      body.classList.remove("is-sidebar-collapsed");
      body.classList.toggle("is-sidebar-open", !!state.mobileOpen);
    } else {
      body.classList.remove("is-sidebar-open");
      body.classList.toggle("is-sidebar-collapsed", !!state.collapsed);
    }
    setSidebarAria();
    setToggleAria();
  };

  const initGroups = (state) => {
    const groups = Array.from(document.querySelectorAll(".nav-group"));
    groups.forEach((group, idx) => {
      const btn = group.querySelector(".nav-group-btn");
      const sub = group.querySelector(".nav-sub");
      if (!btn || !sub) return;

      const key = btn.dataset.key || slugify(btn.textContent) || `group-${idx}`;
      btn.dataset.key = key;

      const hasActive = !!group.querySelector(".nav-sub-item.is-active");
      const serverOpen = group.classList.contains("is-open");
      const storedOpen = !!state.groups[key];

      const shouldOpen = hasActive || serverOpen || storedOpen;
      group.classList.toggle("is-open", shouldOpen);
      btn.setAttribute("aria-expanded", shouldOpen ? "true" : "false");

      btn.addEventListener("click", () => {
        const willOpen = !group.classList.contains("is-open");
        group.classList.toggle("is-open", willOpen);
        btn.setAttribute("aria-expanded", willOpen ? "true" : "false");
        state.groups[key] = willOpen;
        saveState(state);
      });
    });
  };

  const initSidebarToggles = (state) => {
    const overlay = ensureOverlay();

    const closeMobile = () => {
      body.classList.remove("is-sidebar-open");
      state.mobileOpen = false;
      saveState(state);
      setSidebarAria();
      setToggleAria();
    };

    overlay.addEventListener("click", () => {
      if (isMobile()) closeMobile();
    });

    document.addEventListener("keydown", (e) => {
      if (e.key === "Escape" && isMobile() && body.classList.contains("is-sidebar-open")) closeMobile();
    });

    const toggles = Array.from(document.querySelectorAll(".js-sidebar-toggle"));
    toggles.forEach((btn) => {
      btn.addEventListener("click", () => {
        if (isMobile()) {
          const willOpen = !body.classList.contains("is-sidebar-open");
          body.classList.toggle("is-sidebar-open", willOpen);
          state.mobileOpen = willOpen;
          saveState(state);
        } else {
          const isOpener = btn.classList.contains("sidebar-opener");
          const isCloser = btn.classList.contains("sidebar-close");

          let collapsed = body.classList.contains("is-sidebar-collapsed");
          if (isOpener) collapsed = false;
          else if (isCloser) collapsed = true;
          else collapsed = !collapsed;

          body.classList.toggle("is-sidebar-collapsed", collapsed);
          state.collapsed = collapsed;
          saveState(state);
        }

        setSidebarAria();
        setToggleAria();
      });
    });

    window.matchMedia("(max-width: 760px)").addEventListener("change", () => {
      applyLayoutState(state);
    });
  };

  document.addEventListener("DOMContentLoaded", () => {
    const state = loadState();
    applyLayoutState(state);
    initGroups(state);
    initSidebarToggles(state);
  });
})();
