// /home/bungalowsepeti/public_html/panel/js/upload.js
(() => {
  "use strict";

  const qs = (s, r = document) => r.querySelector(s);
  const qsa = (s, r = document) => Array.from(r.querySelectorAll(s));

  const BS = (window.BS = window.BS || {});
  const toast = (msg, type = "info", timeout = 3200) => BS?.toast?.(msg, type, timeout);
  const confirmModal = (opts) => (BS?.confirmModal ? BS.confirmModal(opts) : Promise.resolve(false));
  const lightbox = (src) => BS?.lightbox?.(src);

  /* ----------------------------
     Helpers
  ----------------------------- */
  const escapeHtml = (str) =>
    String(str).replace(/[&<>"']/g, (m) => ({ "&": "&amp;", "<": "&lt;", ">": "&gt;", '"': "&quot;", "'": "&#39;" }[m]));

  const formatKB = (bytes) => {
    const kb = (Number(bytes) || 0) / 1024;
    if (kb < 1024) return `${kb.toFixed(1)} KB`;
    const mb = kb / 1024;
    return `${mb.toFixed(1)} MB`;
  };

  const toIntId = (x) => {
    if (Number.isInteger(x) && x > 0) return x;
    if (typeof x === "number" && Number.isFinite(x)) {
      const n = Math.trunc(x);
      return n > 0 ? n : 0;
    }
    if (typeof x === "string") {
      const t = x.trim();
      if (!/^\d+$/.test(t)) return 0;
      const n = parseInt(t, 10);
      return Number.isInteger(n) && n > 0 ? n : 0;
    }
    return 0;
  };

  const uniqIntIds = (arr) => {
    const out = [];
    const seen = new Set();
    (arr || []).forEach((x) => {
      const id = toIntId(x);
      if (!id) return;
      if (seen.has(id)) return;
      seen.add(id);
      out.push(id);
    });
    return out;
  };

  const csvToIds = (csv) => {
    const s = String(csv || "").trim();
    if (!s) return [];
    return uniqIntIds(
      s
        .split(",")
        .map((p) => p.trim())
        .filter(Boolean)
    );
  };

  // ilan.js gibi dosyalar buradan okuyabilsin
  window.BS_UTIL = Object.assign({}, window.BS_UTIL || {}, { toIntId, uniqIntIds, csvToIds });



  function xhrPost(url, formData, onProgress) {
    return new Promise((resolve, reject) => {
      const xhr = new XMLHttpRequest();
      xhr.open("POST", url, true);

      xhr.upload.onprogress = (evt) => {
        if (!evt.lengthComputable) return;
        const p = Math.round((evt.loaded / evt.total) * 100);
        if (onProgress) onProgress(p);
      };

      xhr.onreadystatechange = () => {
        if (xhr.readyState !== 4) return;

        let data = {};
        try {
          data = JSON.parse(xhr.responseText || "{}");
        } catch {
          // JSON değilse
          data = {};
        }

        if (xhr.status >= 200 && xhr.status < 300) resolve(data);
        else reject(data?.message || "İşlem başarısız.");
      };

      xhr.onerror = () => reject("Bağlantı hatası.");
      xhr.send(formData);
    });
  }

  /* ----------------------------
     ORTAM KÜTÜPHANESİ (ortam-kutuphanesi.php sayfası)
  ----------------------------- */
  function initMediaLibrary() {
    if (initMediaLibrary._inited) return;
    initMediaLibrary._inited = true;

    if (!window.BS_MEDIA) return;

    const endpoint = String(window.BS_MEDIA.endpoint || "");
    const csrf = String(window.BS_MEDIA.csrf || "");

    if (!endpoint || !csrf) return;

    const input = qs("#fileInput");
    const drop = qs("#dropZone");
    const grid = qs("#mediaGrid");
    const totalEl = qs("#mediaTotal");

    const bulkToggle = qs("#bulkToggle");
    const bulkBar = qs("#bulkBar");
    const bulkCancel = qs("#bulkCancel");
    const bulkDelete = qs("#bulkDelete");
    const bulkCount = qs("#bulkCount");

    let selecting = false;
    const selected = new Set();

    const bumpTotal = (delta) => {
      if (!totalEl) return;
      const n = Number(totalEl.textContent || "0") || 0;
      totalEl.textContent = String(Math.max(0, n + delta));
    };

    const setSelecting = (on) => {
      selecting = !!on;
      selected.clear();
      document.body.classList.toggle("is-selecting", selecting);
      if (bulkBar) bulkBar.classList.toggle("is-hidden", !selecting);
      if (bulkCount) bulkCount.textContent = "0";
      if (bulkDelete) bulkDelete.disabled = true;
      if (grid) qsa(".media-card.is-selected", grid).forEach((c) => c.classList.remove("is-selected"));
      if (bulkToggle) bulkToggle.textContent = selecting ? "Seçim Açık" : "Toplu Seçim";
    };

    const updateBulkUI = () => {
      const n = selected.size;
      if (bulkCount) bulkCount.textContent = String(n);
      if (bulkDelete) bulkDelete.disabled = n < 1;
    };

    const toggleCardSelection = (card) => {
      const id = toIntId(card?.dataset?.id);
      if (!id) return;

      if (selected.has(id)) {
        selected.delete(id);
        card.classList.remove("is-selected");
      } else {
        selected.add(id);
        card.classList.add("is-selected");
      }
      updateBulkUI();
    };

    const normalizeFiles = (fileList) => {
      const files = Array.from(fileList || []).filter(Boolean);
      if (files.length === 0) return [];
      if (files.length > 10) {
        toast("Aynı anda en fazla 10 görsel yükleyebilirsin.", "error");
        return files.slice(0, 10);
      }
      return files;
    };

    const makePlaceholder = (file) => {
      const card = document.createElement("div");
      card.className = "media-card is-loading";
      card.dataset.file = file.name;
      card.innerHTML = `
        <div class="media-thumb media-thumb-loading">
          <div class="media-skeleton"></div>
          <div class="media-progress"><div class="media-progress-bar" style="width:0%"></div></div>
          <div class="media-wait">Kaydediliyor...</div>
        </div>
        <div class="media-meta">
          <div class="media-name" title="${escapeHtml(file.name)}">${escapeHtml(file.name)}</div>
          <div class="media-sub muted">0.0 KB</div>
        </div>
        <div class="media-actions">
          <button class="icon-btn" type="button" disabled>⏳</button>
        </div>
      `;
      if (grid) grid.prepend(card);
      return card;
    };

    const uploadOne = async (file, sortKey) => {
      if (!file) return;

      if (file.size > 10 * 1024 * 1024) {
        toast(`${file.name}: 10MB sınırını aşıyor.`, "error");
        return;
      }

      const ph = makePlaceholder(file);
      const bar = qs(".media-progress-bar", ph);
      const sub = qs(".media-sub", ph);

      const fd = new FormData();
      fd.append("csrf_token", csrf);
      fd.append("action", "ajax_upload");
      fd.append("sort_key", String(sortKey));
      fd.append("file", file);

      try {
        const res = await xhrPost(endpoint, fd, (p) => {
          if (bar) bar.style.width = `${p}%`;
          if (sub) sub.textContent = formatKB(file.size * (p / 100));
        });

        if (!res || !res.ok) throw res?.message || "Yükleme başarısız.";

        const id = toIntId(res.id);
        const url = String(res.url || "");
        const name = String(res.file || file.name);
        const size = Number(res.size || file.size || 0);

        ph.classList.remove("is-loading");
        ph.dataset.id = String(id);
        ph.dataset.file = name;
        ph.dataset.src = url;

        ph.innerHTML = `
          <button class="media-thumb js-preview" type="button" aria-label="Önizle">
            <img class="media-img" src="${escapeHtml(url)}" alt="">
          </button>
          <div class="media-meta">
            <div class="media-name" title="${escapeHtml(name)}">${escapeHtml(name)}</div>
            <div class="media-sub muted">${escapeHtml(res.size_kb || formatKB(size))}</div>
          </div>
          <div class="media-actions">
            <button class="icon-btn js-delete" type="button" title="Sil">🗑️</button>
          </div>
        `;

        bumpTotal(1);
      } catch (err) {
        ph.remove();
        toast(typeof err === "string" ? err : "Yükleme başarısız.", "error");
      }
    };

    const uploadFiles = (files) => {
      const list = normalizeFiles(files);
      const base = Date.now() * 1000;
      list.forEach((file, i) => uploadOne(file, base + i));
    };

    if (input) {
      input.addEventListener("change", () => {
        uploadFiles(input.files);
        input.value = "";
      });
    }

    if (drop) {
      let dragDepth = 0;
      const stop = (e) => {
        e.preventDefault();
        e.stopPropagation();
      };
      ["dragenter", "dragover", "dragleave", "drop"].forEach((ev) => drop.addEventListener(ev, stop));

      drop.addEventListener("dragenter", () => {
        dragDepth++;
        drop.classList.add("is-drag");
      });

      drop.addEventListener("dragover", () => drop.classList.add("is-drag"));

      drop.addEventListener("dragleave", () => {
        dragDepth = Math.max(0, dragDepth - 1);
        if (dragDepth === 0) drop.classList.remove("is-drag");
      });

      drop.addEventListener("drop", (e) => {
        dragDepth = 0;
        drop.classList.remove("is-drag");
        uploadFiles(e.dataTransfer.files);
      });
    }

    if (bulkToggle) bulkToggle.addEventListener("click", () => setSelecting(!selecting));
    if (bulkCancel) bulkCancel.addEventListener("click", () => setSelecting(false));

    if (bulkDelete) {
      bulkDelete.addEventListener("click", async () => {
        if (selected.size < 1) return;

        const ok = await confirmModal({
          title: "Kalıcı Olarak Sil",
          text: `${selected.size} görsel kalıcı olarak silinsin mi?`,
          okText: "Evet, Sil",
          cancelText: "Vazgeç",
        });
        if (!ok) return;

        const fd = new FormData();
        fd.append("csrf_token", csrf);
        fd.append("action", "ajax_bulk_delete");
        Array.from(selected).forEach((id) => fd.append("ids[]", String(id)));

        try {
          const res = await xhrPost(endpoint, fd);
          if (!res || !res.ok) throw res?.message || "Silme başarısız.";

          const deleted = Array.isArray(res.deleted_ids) ? res.deleted_ids : [];
          deleted.forEach((id) => {
            const card = grid ? grid.querySelector(`.media-card[data-id="${id}"]`) : null;
            if (card) card.remove();
          });

          bumpTotal(-(deleted.length || 0));
          toast("Silindi.", "success", 2200);
          setSelecting(false);
        } catch (err) {
          toast(typeof err === "string" ? err : "Silme başarısız.", "error");
        }
      });
    }

    // Tek listener ile yönet (document yerine grid üzerinden, ama fallback olarak document da iş görür)
    document.addEventListener("click", async (e) => {
      const card = e.target.closest(".media-card");

      if (selecting && card && !card.classList.contains("is-loading")) {
        if (e.target.closest(".js-delete")) return;
        toggleCardSelection(card);
        return;
      }

      const del = e.target.closest(".js-delete");
      if (del) {
        const c = del.closest(".media-card");
        const id = toIntId(c?.dataset?.id);
        const file = c ? c.dataset.file || "" : "";
        if (!id) return;

        const ok = await confirmModal({
          title: "Görseli Sil",
          text: file ? `${file} silinsin mi?` : "Görsel silinsin mi?",
          okText: "Evet, Sil",
          cancelText: "Vazgeç",
        });
        if (!ok) return;

        const fd = new FormData();
        fd.append("csrf_token", csrf);
        fd.append("action", "ajax_delete");
        fd.append("id", String(id));

        try {
          const res = await xhrPost(endpoint, fd);
          if (!res || !res.ok) throw res?.message || "Silme başarısız.";
          if (c) c.remove();
          bumpTotal(-1);
          toast("Silindi.", "success", 2200);
        } catch (err) {
          toast(typeof err === "string" ? err : "Silme başarısız.", "error");
        }
        return;
      }

      const prev = e.target.closest(".js-preview");
      if (prev) {
        const c = prev.closest(".media-card");
        const src = c?.dataset?.src || "";
        if (src) lightbox(src);
      }
    });
  }

  /* ----------------------------
     İLAN: KAPAK / GALERİ SEÇİCİ (ilan-ekle.php / ilan-duzenle.php)
  ----------------------------- */
  function initIlanMediaPicker() {
    if (initIlanMediaPicker._inited) return;
    initIlanMediaPicker._inited = true;

    const coverInp = qs("#cover_media_id");
    const galInp = qs("#gallery_media_ids");
    const coverBox = qs("#pickedCover");
    const galBox = qs("#pickedGallery");
    const btnCover = qs("#btnPickCover");
    const btnGal = qs("#btnPickGallery");

    if (!coverInp || !galInp || (!btnCover && !btnGal)) return;

    const csrf =
      (window.BS_UPLOAD && String(window.BS_UPLOAD.csrf || "")) || String(qs('input[name="csrf_token"]')?.value || "");

    const listEndpoint =
      (window.BS_UPLOAD && String(window.BS_UPLOAD.listEndpoint || window.BS_UPLOAD.ilanEndpoint || "")) ||
      (location.pathname.split("/").pop() || "");

    const uploadEndpoint =
      (window.BS_UPLOAD && String(window.BS_UPLOAD.uploadEndpoint || "")) || "ortam-kutuphanesi.php";

    const maxGallery = Number((window.BS_UPLOAD && window.BS_UPLOAD.maxGallery) || 10) || 10;

    const mediaSection = qs('[data-req="media"]');
    const previewWrap = mediaSection ? qs(".media-preview", mediaSection) : null;
    const coverCol = coverBox ? coverBox.closest(".media-preview-col") : null;
    const galCol = galBox ? galBox.closest(".media-preview-col") : null;

    const state = {
      coverId: toIntId(coverInp.value),
      galleryIds: csvToIds(galInp.value),
    };

    // başlangıç normalize
    state.galleryIds = uniqIntIds(state.galleryIds)
      .filter((id) => id !== state.coverId)
      .slice(0, maxGallery);

    const updatePreviewUI = () => {
      const hasCover = !!state.coverId;
      const hasGal = (state.galleryIds || []).length > 0;
      const hasAny = hasCover || hasGal;

      if (previewWrap) previewWrap.style.display = hasAny ? "" : "none";
      if (coverCol) coverCol.style.display = hasAny ? "" : "none";
      if (galCol) galCol.style.display = hasGal ? "" : "none";
      if (coverCol) coverCol.style.gridColumn = hasCover && !hasGal ? "1 / -1" : "";
      if (galCol) galCol.style.gridColumn = "";
    };

    const saveHidden = () => {
      state.coverId = toIntId(state.coverId);

      state.galleryIds = uniqIntIds(state.galleryIds)
        .filter((id) => id !== state.coverId)
        .slice(0, maxGallery);

      coverInp.value = state.coverId ? String(state.coverId) : "";
      galInp.value = state.galleryIds.length ? state.galleryIds.join(",") : "";

      // ilan.js tarafında kart durumlarını güncelle
      BS?.updateIlanCards?.();
    };

    const fetchMediaList = async () => {
      if (!csrf || !listEndpoint) throw "CSRF / endpoint eksik.";

      const fd = new FormData();
      fd.append("csrf_token", csrf);
      fd.append("action", "ajax_media_list");

      const res = await fetch(listEndpoint, {
        method: "POST",
        body: fd,
        cache: "no-store",
        credentials: "same-origin",
      });

      let data = {};
      try {
        data = await res.json();
      } catch {
        data = {};
      }

      if (!res.ok || !data.ok) throw data?.message || "Liste alınamadı.";
      return Array.isArray(data.items) ? data.items : [];
    };

    let cacheMap = null;

    const fetchMediaMap = async () => {
      if (cacheMap) return cacheMap;
      const items = await fetchMediaList();
      const m = new Map();
      items.forEach((it) => m.set(Number(it.id), it));
      cacheMap = m;
      return m;
    };

    const renderCover = async () => {
      if (!coverBox) return;
      coverBox.innerHTML = "";
      if (!state.coverId) return;

      const map = await fetchMediaMap();
      const it = map.get(state.coverId);

      // Seçili kapak artık yoksa temizle
      if (!it) {
        state.coverId = 0;
        saveHidden();
        return;
      }

      const card = document.createElement("div");
      card.className = "media-card";
      card.dataset.id = String(state.coverId);
      card.dataset.src = String(it.url || "");
      card.innerHTML = `
        <button class="media-thumb js-preview" type="button" aria-label="Önizle">
          <img class="media-img" src="${escapeHtml(it.url)}" alt="">
        </button>
        <div class="media-actions" style="left:8px;left:auto;">
          <button class="icon-btn js-clear-cover" type="button" title="Kapak Kaldır">✕</button>
        </div>
      `;
      coverBox.appendChild(card);
    };

    const enableDragSort = () => {
      if (!galBox) return;
      if (enableDragSort._inited) return;
      enableDragSort._inited = true;

      let dragEl = null;
      let placeholder = null;
      let lastKey = "";

      const ensurePlaceholder = (likeEl) => {
        if (placeholder) return;
        placeholder = document.createElement("div");
        placeholder.className = "media-card bs-sort-placeholder";
        const r = likeEl.getBoundingClientRect();
        placeholder.style.height = `${Math.max(1, r.height)}px`;
      };

      const placePlaceholder = (refNode, after = false) => {
        if (!placeholder) return;
        const ref = after ? refNode?.nextSibling : refNode;
        if (ref === placeholder) return;
        galBox.insertBefore(placeholder, ref || null);
      };

      const buildRows = () => {
        const cards = qsa(".js-sort-card", galBox).filter((c) => c !== dragEl && c !== placeholder);

        const items = cards
          .map((el) => ({ el, r: el.getBoundingClientRect() }))
          .sort((a, b) => a.r.top - b.r.top || a.r.left - b.r.left);

        const tol = 12;
        const rows = [];
        items.forEach((it) => {
          let row = rows.find((x) => Math.abs(x.top - it.r.top) <= tol);
          if (!row) {
            row = { top: it.r.top, height: it.r.height, items: [] };
            rows.push(row);
          }
          row.items.push(it);
          row.height = Math.max(row.height, it.r.height);
        });

        rows.forEach((row) => row.items.sort((a, b) => a.r.left - b.r.left));
        return rows;
      };

      const getInsertPoint = (x, y) => {
        const rows = buildRows();
        if (!rows.length) return { ref: null, after: false };

        let row = rows.find((r) => y >= r.top && y <= r.top + r.height);
        if (!row) {
          row = rows.reduce(
            (best, r) => {
              const mid = r.top + r.height / 2;
              const d = Math.abs(y - mid);
              return d < best.d ? { r, d } : best;
            },
            { r: rows[0], d: Infinity }
          ).r;
        }

        let ref = null;
        let after = false;

        for (const it of row.items) {
          const midX = it.r.left + it.r.width / 2;
          if (x < midX) {
            ref = it.el;
            after = false;
            break;
          }
        }

        if (!ref) {
          const last = row.items[row.items.length - 1];
          ref = last?.el || null;
          after = true;
        }

        return { ref, after };
      };

      const finalize = () => {
        if (!dragEl || !placeholder) return;

        galBox.insertBefore(dragEl, placeholder);
        placeholder.remove();

        dragEl.style.display = "";
        dragEl.classList.remove("is-dragging");

        const ids = [];
        qsa(".js-sort-card", galBox).forEach((c) => {
          const id = toIntId(c.dataset.id);
          if (id) ids.push(id);
        });

        state.galleryIds = ids;
        saveHidden();

        dragEl = null;
        placeholder = null;
        lastKey = "";
      };

      galBox.addEventListener("dragstart", (e) => {
        const c = e.target.closest(".js-sort-card");
        if (!c) return;

        const id = toIntId(c.dataset.id);
        if (!id) return;

        dragEl = c;

        ensurePlaceholder(c);
        c.classList.add("is-dragging");

        galBox.insertBefore(placeholder, c.nextSibling);

        setTimeout(() => {
          if (dragEl) dragEl.style.display = "none";
        }, 0);

        try {
          e.dataTransfer.effectAllowed = "move";
          e.dataTransfer.setData("text/plain", String(id));
        } catch {}
      });

      galBox.addEventListener("dragover", (e) => {
        if (!dragEl || !placeholder) return;
        e.preventDefault();

        const { ref, after } = getInsertPoint(e.clientX, e.clientY);
        const key = `${ref ? ref.dataset.id : "end"}:${after ? "a" : "b"}`;
        if (key === lastKey) return;
        lastKey = key;

        if (!ref) placePlaceholder(null, false);
        else placePlaceholder(ref, after);
      });

      galBox.addEventListener("drop", (e) => {
        if (!dragEl) return;
        e.preventDefault();
        finalize();
      });

      galBox.addEventListener("dragend", () => finalize());
    };

    const renderGallery = async () => {
      if (!galBox) return;
      galBox.innerHTML = "";
      if (!state.galleryIds.length) return;

      const map = await fetchMediaMap();

      // Listedeki olmayan id’leri temizle
      const alive = [];
      state.galleryIds.forEach((id) => {
        const it = map.get(id);
        if (!it) return;
        alive.push(id);

        const card = document.createElement("div");
        card.className = "media-card js-sort-card";
        card.draggable = true;
        card.dataset.id = String(id);
        card.dataset.src = String(it.url || "");

        card.innerHTML = `
          <button class="media-thumb js-preview" type="button" aria-label="Önizle">
            <img class="media-img" src="${escapeHtml(it.url)}" alt="">
          </button>
          <div class="media-actions" style="left:8px;left:auto;">
            <button class="icon-btn js-remove" type="button" title="Kaldır">✕</button>
          </div>
        `;

        galBox.appendChild(card);
      });

      if (alive.length !== state.galleryIds.length) {
        state.galleryIds = alive;
        saveHidden();
      }

      enableDragSort();
    };

    const renderAll = async () => {
      saveHidden();
      updatePreviewUI();

      // boşsa fetch yapma
      if (!state.coverId && !state.galleryIds.length) {
        if (coverBox) coverBox.innerHTML = "";
        if (galBox) galBox.innerHTML = "";
        return;
      }

      // sadece ihtiyaç varsa fetch
      if (state.coverId) await renderCover();
      else if (coverBox) coverBox.innerHTML = "";

      if (state.galleryIds.length) await renderGallery();
      else if (galBox) galBox.innerHTML = "";
    };

       const openPicker = async (mode) => {
      if (!window.BS?.openMediaPicker) {
        toast("modal.js yüklenmemiş.", "error");
        return;
      }

      let result = null;
      try {
        result = await window.BS.openMediaPicker({
          mode,
          csrf,
          uploadEndpoint,
          maxGallery,
          coverId: state.coverId,
          pickedIds: mode === "cover" ? [state.coverId] : state.galleryIds,
          fetchItems: fetchMediaList,
          xhrPost,
          toast,
        });
      } catch (err) {
        toast(String(err || "İşlem başarısız."), "error");
        return;
      }

      if (!result) return;

      if (mode === "cover") {
        const id = toIntId(result.coverId);
        if (!id) return;
        state.coverId = id;
        state.galleryIds = state.galleryIds.filter((x) => x !== id);
        await renderAll();
        return;
      }

      const ids = uniqIntIds(result.galleryIds || []);
      state.galleryIds = ids.filter((id) => id !== state.coverId).slice(0, maxGallery);
      await renderAll();
    };


    btnCover?.addEventListener("click", () => openPicker("cover"));
    btnGal?.addEventListener("click", () => openPicker("gallery"));

    document.addEventListener("click", (e) => {
      const rm = e.target.closest(".js-remove");
      if (rm) {
        const card = rm.closest(".media-card");
        const id = toIntId(card?.dataset?.id);
        if (!id) return;
        state.galleryIds = state.galleryIds.filter((x) => x !== id);
        renderAll().catch(() => {});
        return;
      }

      const clearCover = e.target.closest(".js-clear-cover");
      if (clearCover) {
        state.coverId = 0;
        renderAll().catch(() => {});
        return;
      }

      const prev = e.target.closest(".js-preview");
      if (prev) {
        const card = prev.closest(".media-card");
        const src = card?.dataset?.src || "";
        if (src) lightbox(src);
      }
    });

    renderAll().catch(() => {});
  }

  document.addEventListener("DOMContentLoaded", () => {
    initMediaLibrary();
    initIlanMediaPicker();
  });

  window.BS_UPLOAD = Object.assign({}, window.BS_UPLOAD || {}, {
    initMediaLibrary,
    initIlanMediaPicker,
  });
})();
